<?PHP
#
#   FILE:  Results.php
#
#   FUNCTIONS PROVIDED:
#       PrintLimits($Limits, $Limit)
#           - Print each limit
#       PrintPageLinks($NumPages, $Page)
#           - Print the previous and next page links
#       PrintInvalidResourceUrls($InvalidUrls)
#           - Print the data values for each invalid URL
#       PrintActions($Values)
#           - Print the actions that should be displayed for each failure
#       safehtmlentities($string, $quote_style, $charset, $double_encode)
#           - Version-agnostic htmlentities() function
#
#   FUNCTIONS EXPECTED:
#       PrintPreviousLink($Page)
#           - Print the link to go to the previous page
#       PrintNextLink($Page)
#           - Print the link to go to the next page, if not the second to last
#       PrintLastLink($Page)
#           - Print the link to go to the last page
#       PrintSelectedLimit($Limit)
#           - Print a limit that is selected, i.e., the current limit
#       PrintLimit($Limit)
#           - Print a limit that is not selected, i.e., not the current limit
#       PrintInvalidResourceUrl($Values)
#           - Print out a single invalid resource URL's data
#       PrintEdit($Values)
#           - Print out the edit action. This is always displayed
#       PrintEnable($Values)
#           - Print out the action associated with enabling the given resource
#       PrintDisable($Values)
#           - Print out the action associated with disabling the given resource
#
#   OPTIONAL FUNCTIONS:
#       (none)
#
#   FORM FIELDS EXPECTED:
#       (none)
#
#   Copyright 2009 Internet Scout Project
#   http://scout.wisc.edu/
#

# ----- EXPORTED FUNCTIONS ---------------------------------------------------

/**
 * Print each limit.
 * @param $Limits array of limit values to print
 * @param $Limit the selected limit
 */
function PrintLimits($Limits, $Limit)
{
    foreach ($Limits as $TempLimit)
    {
        # if the temp limit is the same as the current one
        if ($TempLimit == $Limit)
        {
            PrintSelectedLimit($TempLimit);
        }

        # any other limit
        else
        {
            PrintLimit($TempLimit);
        }
    }
}

/**
 * Print the previous and next page links.
 * @param $NumPages total number of pages
 * @param $Page current page number
 * @param $Limit total URLs per page to display
 */
function PrintPageLinks($NumPages, $Page, $Limit)
{
    # previous urls exist
    if ($Page > 1)
    {
        PrintPreviousLink($Limit);
    }

    # next urls exist and not the second to last page
    if ($Page < $NumPages - 1)
    {
        PrintNextLink($Limit);
    }

    # next urls exist and the second to last page
    else if ($Page < $NumPages)
    {
        PrintLastLink($Limit);
    }
}

/**
 * Print the data values for each invalid URL.
 * @param $InvalidUrls array of data values for the invalid URLs
 */
function PrintInvalidResourceUrls($InvalidUrls)
{
    foreach ($InvalidUrls as $Values)
    {
        PrintInvalidResourceUrl($Values);
    }
}

/**
 * Print the actions that should be displayed for each failure.
 * @param $Values values of the failure record
 */
function PrintActions($Values)
{
    # always print the edit action
    PrintEdit($Values);

    # additional actions based on the http status code
    switch ($Values["StatusNo"])
    {
        case 0:
        case 404:
            # already disabled, re-enable
            $Resource = new Resource($Values["ResourceId"]);
            if ($Resource->Status() && !intval($Resource->Get("Release Flag")))
            {
                PrintEnable($Values);
            }

            # disable
            else
            {
                PrintDisable($Values);
            }
            break;
    }
}

/**
 * Version-agnostic htmlentities() function.
 * See: http://www.php.net/manual/en/function.htmlentities.php
 * @param $string the input string
 * @param $quote_style defines what is done with quotes
 * @param $charset defines the character set used in conversion
 * @param $double_encode when off, PHP won't encode existing HTML entities (>= 5.2.3)
 * @return the encoded string
 */
function safehtmlentities($string, $quote_style=ENT_COMPAT, $charset="ISO-8859-1", $double_encode=TRUE)
{
    # get php's version
    list($Major, $Minor, $Maint) = explode(".", phpversion());
    $Major = intval($Major);
    $Minor = intval($Minor);
    $Maint = intval($Maint);

    # ignore $double_encode param if php version < 5.2.3
    if ($Major < 5 || $Major < 6 && $Minor < 2 || $Major < 6 && $Minor < 3 && $Maint < 4)
    {
        return htmlentities($string, $quote_style, $charset);
    }

    return htmlentities($string, $quote_style, $charset, $double_encode);
}

# ----- LOCAL FUNCTIONS ------------------------------------------------------

/**
 * Get the number of URL-typed fields.
 * @return the number of URL-typed fields
 */
function GetNumUrlFields()
{
    $Schema = new MetadataSchema();
    return count($Schema->GetFields(MetadataSchema::MDFTYPE_URL));
}

# ----- MAIN -----------------------------------------------------------------

# non-standard globals
global $PageName;
global $UpdatePageName;
global $Stats;
global $NumUrlFields;
global $Offset;
global $Limit;
global $ReleaseFlag;
global $FailureSum;
global $Failures;
global $Page;
global $NumPages;
global $Limits;

# set the title
PageTitle("URL Checker Results");

# check if current user is authorized
CheckAuthorization(PRIV_SYSADMIN, PRIV_COLLECTIONADMIN);

# page information
$PageName = (isset($_GET["P"])) ? $_GET["P"] : "";
$UpdatePageName = "P_UrlChecker_Update";

# remove stale data and get stats
$AF->SignalEvent("URLCHECKER_REMOVE_STALE_DATA");
$Stats = $AF->SignalEvent("URLCHECKER_GET_STATS");
$NumUrlFields = GetNumUrlFields();

# limits on which resources are displayed
$Offset = (isset($_GET["Offset"])) ? intval($_GET["Offset"]) : 0;
$Limit = (isset($_GET["Limit"])) ? intval($_GET["Limit"]) : 15;
$ReleaseFlag = (isset($_GET["ReleaseFlag"])) ? intval($_GET["ReleaseFlag"]) : 1;

# assume we'll fail to get these
$FailureSum = 0;
$Failures = array();

# get invalid urls for released resources only
if ($ReleaseFlag)
{
    $FailureSum = $Stats["NumReleasedFailures"];
    $Failures = $AF->SignalEvent("URLCHECKER_GET_RELEASED_FAILURES", array($Limit, $Offset));
}

# get unreleased ones
else
{
    $FailureSum = $Stats["NumUnreleasedFailures"];
    $Failures = $AF->SignalEvent("URLCHECKER_GET_UNRELEASED_FAILURES", array($Limit, $Offset));
}

# pagination info
$Page = ($Limit > 0 && $Offset > 0) ? ceil($Offset/$Limit)+1 : 1;
$NumPages = ($Limit > 0) ? ceil($FailureSum/$Limit) : 1;
$Limits = array(15, 25, 50, 100);

?>
