<?PHP
#
#   FILE:  MetricsReporter.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2011 Edward Almasy and Internet Scout
#   http://scout.wisc.edu
#

class MetricsReporter extends Plugin {

    function Register()
    {
        $this->Name = "Metrics Reporter";
        $this->Version = "0.9.1";
        $this->Description = "Generates usage and web metrics reports"
                ." from data recorded by the <i>Metrics Recorder</i> plugin.";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = array(
                "CWISCore" => "2.2.3",
                "MetricsRecorder" => "1.1.3");
        $this->EnabledByDefault = FALSE;

        $this->CfgSetup["PrivsToExcludeFromCounts"] = array(
                "Type" => "Privileges",
                "Label" => "Exclude Users with Any of These Privileges",
                "Help" => "Users with these privilege flags will be"
                        ." excluded from running full record view count"
                        ." and URL click count figures.",
                );
    }

    function Install()
    {
        $this->ConfigSetting("PrivsToExcludeFromCounts", array(
                PRIV_SYSADMIN,
                PRIV_RESOURCEADMIN,
                PRIV_CLASSADMIN,
                PRIV_NAMEADMIN,
                PRIV_RELEASEADMIN,
                PRIV_COLLECTIONADMIN,
                ));
    }

    function Initialize()
    {
        $Schema = new MetadataSchema();
        if (!$Schema->FieldExists("Full Record View Count"))
        {
            $ViewCountFieldDescription = <<<EOT
                <Owner>MetricsReporter</Owner>
                <Name>Full Record View Count</Name>
                <Type>MDFTYPE_NUMBER</Type>
                <Label>Full Record Views</Label>
                <Description>This field is used by the Metrics Reporter plugin
                    to store a count of the number of times the full record page
                    for a resource has been viewed by non-privileged users.</Description>
                <Editable>FALSE</Editable>
                <Enabled>TRUE</Enabled>
                <ViewingPrivilege>PRIV_COLLECTIONADMIN</ViewingPrivilege>
                <DefaultValue>-1</DefaultValue>
EOT;
            $Field = $Schema->AddFieldFromXml($ViewCountFieldDescription);
        }
        if (!$Schema->FieldExists("URL Field Click Count"))
        {
            $ViewCountFieldDescription = <<<EOT
                <Owner>MetricsReporter</Owner>
                <Name>URL Field Click Count</Name>
                <Type>MDFTYPE_NUMBER</Type>
                <Label>Resource URL Clicks</Label>
                <Description>This field is used by the Metrics Reporter plugin
                    to store a count of the number of times the primary URL for
                    a resource has been clicked by non-privileged users.</Description>
                <Editable>FALSE</Editable>
                <Enabled>TRUE</Enabled>
                <ViewingPrivilege>PRIV_COLLECTIONADMIN</ViewingPrivilege>
                <DefaultValue>-1</DefaultValue>
EOT;
            $Field = $Schema->AddFieldFromXml($ViewCountFieldDescription);
        }
    }

    function HookEvents()
    {
        return array(
                "EVENT_COLLECTION_ADMINISTRATION_MENU" => "AddCollectionAdminMenuItems",
                "EVENT_FULL_RECORD_VIEW" => "UpdateFullRecordViewCount",
                "EVENT_URL_FIELD_CLICK" => "UpdateUrlFieldClickCount",
                );
    }

    function AddCollectionAdminMenuItems()
    {
        return array(
                "CollectionReports" => "Collection Usage Metrics",
                );
    }

    function UpdateFullRecordViewCount($ResourceId)
    {
        # initialize count if needed
        $Resource = new Resource($ResourceId);
        $this->InitializeCounts($Resource);

        # if user is non-privileged
        if (!$GLOBALS["G_User"]->HasPriv(
                $this->ConfigSetting("PrivsToExcludeFromCounts")))
        {
            # update view count for resource
            $Resource->Set("Full Record View Count", 
                    $Resource->Get("Full Record View Count") + 1);
        }
    }

    function UpdateUrlFieldClickCount($ResourceId, $FieldId)
    {
        # initialize count if needed
        $Resource = new Resource($ResourceId);
        $this->InitializeCounts($Resource);

        # if user is non-privileged 
        if (!$GLOBALS["G_User"]->HasPriv(
                $this->ConfigSetting("PrivsToExcludeFromCounts")))
        {
            # if this is primary resource URL
            $Schema = new MetadataSchema();
            if ($FieldId == $Schema->StdNameToFieldMapping("URL"))
            {
                # update view count for resource
                $Resource->Set("URL Field Click Count", 
                        $Resource->Get("URL Field Click Count") + 1);
            }
        }
    }

    private function InitializeCounts($Resource)
    {
        # if view count has not yet been initialized for resource
        if ($Resource->Get("Full Record View Count") == -1)
        {
            # calculate and store current view count for resource
            $Recorder = $GLOBALS["G_PluginManager"]->GetPlugin("MetricsRecorder");
            $CurrentCount = $Recorder->GetFullRecordViewCount(
                    $Resource->Id(), $this->ConfigSetting("PrivsToExcludeFromCounts"));
            $Resource->Set("Full Record View Count", $CurrentCount);
        }

        # if click count has not yet been initialized for resource
        if ($Resource->Get("URL Field Click Count") == -1)
        {
            # calculate and store current click count for resource
            $Recorder = $GLOBALS["G_PluginManager"]->GetPlugin("MetricsRecorder");
            $Schema = new MetadataSchema();
            $FieldId = $Schema->StdNameToFieldMapping("URL");
            $CurrentCount = $Recorder->GetUrlFieldClickCount(
                    $Resource->Id(), $FieldId, 
                    $this->ConfigSetting("PrivsToExcludeFromCounts"));
            $Resource->Set("URL Field Click Count", $CurrentCount);
        }
    }
}

?>
