<?PHP
#
#   FILE:  BatchEdit.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2014 Internet Scout Project
#   http://scout.wisc.edu/cwis/
#

class BatchEdit extends Plugin
{
    /**
    * Register information about this plugin.
    */
    public function Register()
    {
        $this->Name = "Batch Editing";
        $this->Version = "1.0.1";
        $this->Description = "Allows resources in a folder to be edited en masse.";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = array(
            "CWISCore" => "2.4.1",
            "Folders" => "1.0.7"
            );
        $this->EnabledByDefault = TRUE;

        # fields that can be batch edited
        $FieldTypes =  MetadataSchema::MDFTYPE_TEXT |
                    MetadataSchema::MDFTYPE_PARAGRAPH |
                    MetadataSchema::MDFTYPE_NUMBER |
                    MetadataSchema::MDFTYPE_DATE |
                    MetadataSchema::MDFTYPE_TIMESTAMP |
                    MetadataSchema::MDFTYPE_FLAG |
                    MetadataSchema::MDFTYPE_TREE |
                    MetadataSchema::MDFTYPE_CONTROLLEDNAME |
                    MetadataSchema::MDFTYPE_OPTION |
                    MetadataSchema::MDFTYPE_URL |
                    MetadataSchema::MDFTYPE_REFERENCE;
        $AllowedFields = array();

        # iterate over all schemas, constructing a list of editable
        # fields in each
        $AllSchemas = MetadataSchema::GetAllSchemas();
        foreach ($AllSchemas as $Schema)
        {
            # don't allow batch editing of user fields
            if ($Schema->Id() == MetadataSchema::SCHEMAID_USER)
            {
                continue;
            }

            # add a prefix for schemas that aren't the resource schema
            $Pfx = ($Schema->Id() == MetadataSchema::SCHEMAID_DEFAULT) ?
                 "" : $Schema->Name().": ";
            $SchemaFields = $Schema->GetFields($FieldTypes);
            foreach ($SchemaFields as $Field)
            {
                if ($Field->Editable())
                {
                    $AllowedFields[$Field->Id()] = $Pfx.$Field->Name();
                }
            }
        }

        $this->CfgSetup["AllowedFields"] = array(
            "Type" => "Option",
            "Label" => "Allowed Fields",
            "Help" => "Fields allowed for bulk editing",
            "AllowMultiple" => TRUE,
            "Options" => $AllowedFields,
            "Rows" => count($AllowedFields),
            "Default" => array(),
            );

        $this->CfgSetup["RequiredPrivs"] = array(
            "Type" => "Privileges",
            "Label" => "Required Privileges",
            "AllowMultiple" => TRUE,
            "Help" => "Users with any of the selected privileges will "
                      ."be able to perform batch edits on fields they are otherwise "
                      ."able to edit (i.e. only those they could edit from the "
                      ."Edit Resource page)",
            "Default" => array( PRIV_SYSADMIN ),
            );
    }

    /**
    * Upgrade from a previous version.
    * @param string $PreviousVersion Previous version of the plugin.
    * @return Returns NULL on success and an error message otherwise.
    */
    public function Upgrade($PreviousVersion)
    {
        # upgrade from versions < 1.0.1 to 1.0.1
        if (version_compare($PreviousVersion, "1.0.1", "<"))
        {
            if (is_array($this->ConfigSetting("RequiredPrivs")))
            {
                $RequiredPrivs = new PrivilegeSet();
                $RequiredPrivs->AddPrivilege($this->ConfigSetting("RequiredPrivs"));
                $this->ConfigSetting("RequiredPrivs", $RequiredPrivs);
            }
        }
        return NULL;
    }

    /**
    * Hook the events into the application framework.
    * @return an array of events to be hooked into the application framework
    */
    public function HookEvents()
    {
        return array(
            "EVENT_HTML_INSERTION_POINT" => "InsertButton",
            );
    }

    /**
    * HTML insertion point handler, used to add a 'bulk edit' button
    * to the manage folders page.
    * @param string $PageName Name of currently loaded page.
    * @param string $Location Location where HTML can be inserted.
    * @param mixed $Context Context containing page-defined context data.
    */
    public function InsertButton($PageName, $Location, $Context=NULL)
    {
        if ($Location == "Folder Buttons")
        {
            if ($GLOBALS["G_User"]->IsLoggedIn() &&
                $GLOBALS["G_User"]->HasPriv($this->ConfigSetting("RequiredPrivs")) )
            {
                $Folder = new Folders_Folder($Context["FolderId"]);

                if ($Folder->GetItemCount()>0)
                {
                    if ($PageName == "P_Folders_ManageFolders")
                    {
                        print('<a href="index.php?P=P_BatchEdit_Edit&amp;FI='
                              .$Context["FolderId"]
                              .'" class="cw-button cw-button-constrained '
                              .'cw-button-elegant bulk-edit-button"'
                              .'>Batch Edit</a>');
                    }
                    else if ($PageName == "P_Folders_ViewFolder")
                    {
                        print('<a href="index.php?P=P_BatchEdit_Edit&amp;FI='
                              .$Context["FolderId"]
                              .'" class="cw-button cw-button-elegant '
                              .'bulk-edit-button">Batch Edit</a>');
                    }
                }
            }
        }
    }
}
