<?PHP
#
#   FILE:  AFTaskManager.php
#
#   Part of the ScoutLib application support library
#   Copyright 2009-2018 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu
#

/**
* Task manager component of top-level framework for web applications.
* \nosubgrouping
*/
class AFTaskManager
{
    # ---- PUBLIC INTERFACE --------------------------------------------------

    /** @cond */
    /**
    * Object constructor.
    **/
    public function __construct()
    {
        # set up our internal environment
        $this->DB = new Database();

        # load our settings from database
        $this->LoadSettings();
    }
    /** @endcond */

    /**  Highest priority. */
    const PRIORITY_HIGH = 1;
    /**  Medium (default) priority. */
    const PRIORITY_MEDIUM = 2;
    /**  Lower priority. */
    const PRIORITY_LOW = 3;
    /**  Lowest priority. */
    const PRIORITY_BACKGROUND = 4;

    /**
    * Add task to queue.  The Callback parameters is the PHP "callback" type.
    * If $Callback refers to a function (rather than an object method) that function
    * must be available in a global scope on all pages or must be loadable by
    * ApplicationFramework::LoadFunction().
    * If $Priority is out-of-bounds, it wil be normalized to be within bounds.
    * @param callback $Callback Function or method to call to perform task.
    * @param array $Parameters Array containing parameters to pass to function or
    *       method.  (OPTIONAL, pass NULL for no parameters)
    * @param int $Priority Priority to assign to task.  (OPTIONAL, defaults
    *       to PRIORITY_LOW)
    * @param string $Description Text description of task.  (OPTIONAL)
    */
    public function QueueTask($Callback, $Parameters = NULL,
            $Priority = self::PRIORITY_LOW, $Description = "")
    {
        # make sure priority is within bounds
        $Priority = min(self::PRIORITY_BACKGROUND,
                max(self::PRIORITY_HIGH, $Priority));

        # pack task info and write to database
        if ($Parameters === NULL) {  $Parameters = array();  }
        $this->DB->Query("INSERT INTO TaskQueue"
                ." (Callback, Parameters, Priority, Description)"
                ." VALUES ('".addslashes(serialize($Callback))."', '"
                .addslashes(serialize($Parameters))."', ".intval($Priority).", '"
                .addslashes($Description)."')");
    }

    /**
    * Add task to queue if not already in queue or currently running.
    * If task is already in queue with a lower priority than specified, the task's
    * priority will be increased to the new value.
    * The Callback parameter is the PHP "callback" type.
    * If $Callback refers to a function (rather than an object method) that function
    * must be available in a global scope on all pages or must be loadable by
    * ApplicationFramework::LoadFunction().
    * If $Priority is out-of-bounds, it wil be normalized to be within bounds.
    * @param callback $Callback Function or method to call to perform task.
    * @param array $Parameters Array containing parameters to pass to function or
    *       method.  (OPTIONAL, pass NULL for no parameters)
    * @param int $Priority Priority to assign to task.  (OPTIONAL, defaults
    *       to PRIORITY_LOW)
    * @param string $Description Text description of task.  (OPTIONAL)
    * @return bool TRUE if task was added, otherwise FALSE.
    * @see AFTaskManager::TaskIsInQueue()
    */
    public function QueueUniqueTask($Callback, $Parameters = NULL,
            $Priority = self::PRIORITY_LOW, $Description = "")
    {
        if ($this->TaskIsInQueue($Callback, $Parameters))
        {
            $QueryResult = $this->DB->Query("SELECT TaskId,Priority FROM TaskQueue"
                    ." WHERE Callback = '".addslashes(serialize($Callback))."'"
                    .($Parameters ? " AND Parameters = '"
                            .addslashes(serialize($Parameters))."'" : ""));
            if ($QueryResult !== FALSE)
            {
                # make sure priority is within bounds
                $Priority = min(self::PRIORITY_BACKGROUND,
                        max(self::PRIORITY_HIGH, $Priority));

                $Record = $this->DB->FetchRow();
                if ($Record["Priority"] > $Priority)
                {
                    $this->DB->Query("UPDATE TaskQueue"
                            ." SET Priority = ".intval($Priority)
                            ." WHERE TaskId = ".intval($Record["TaskId"]));
                }
            }
            return FALSE;
        }
        else
        {
            $this->QueueTask($Callback, $Parameters, $Priority, $Description);
            return TRUE;
        }
    }

    /**
    * Check if task is already in queue or currently running.
    * When no $Parameters value is specified the task is checked against
    * any other entries with the same $Callback.
    * @param callback $Callback Function or method to call to perform task.
    * @param array $Parameters Array containing parameters to pass to function or
    *       method.  (OPTIONAL)
    * @return bool TRUE if task is already in queue, otherwise FALSE.
    */
    public function TaskIsInQueue($Callback, $Parameters = NULL)
    {
        $QueuedCount = $this->DB->Query(
                "SELECT COUNT(*) AS FoundCount FROM TaskQueue"
                ." WHERE Callback = '".addslashes(serialize($Callback))."'"
                .($Parameters ? " AND Parameters = '"
                        .addslashes(serialize($Parameters))."'" : ""),
                "FoundCount");
        $RunningCount = $this->DB->Query(
                "SELECT COUNT(*) AS FoundCount FROM RunningTasks"
                ." WHERE Callback = '".addslashes(serialize($Callback))."'"
                .($Parameters ? " AND Parameters = '"
                        .addslashes(serialize($Parameters))."'" : ""),
                "FoundCount");
        $FoundCount = $QueuedCount + $RunningCount;
        return ($FoundCount ? TRUE : FALSE);
    }

    /**
    * Retrieve current number of tasks in queue.
    * @param int $Priority Priority of tasks.  (OPTIONAL, defaults to all priorities)
    * @return int Number of tasks currently in queue.
    */
    public function GetTaskQueueSize($Priority = NULL)
    {
        return $this->GetQueuedTaskCount(NULL, NULL, $Priority);
    }

    /**
    * Retrieve list of tasks currently in queue.
    * @param int $Count Number to retrieve.  (OPTIONAL, defaults to 100)
    * @param int $Offset Offset into queue to start retrieval.  (OPTIONAL)
    * @return array Array with task IDs for index and task info for values.  Task info
    *       is stored as associative array with "Callback" and "Parameter" indices.
    */
    public function GetQueuedTaskList($Count = 100, $Offset = 0)
    {
        return $this->GetTaskList("SELECT * FROM TaskQueue"
                ." ORDER BY Priority, TaskId ", $Count, $Offset);
    }

    /**
    * Get number of queued tasks that match supplied values.  Tasks will
    * not be counted if the values do not match exactly, so callbacks with
    * methods for different objects (even of the same class) will not match.
    * @param callback $Callback Function or method to call to perform task.
    *       (OPTIONAL)
    * @param array $Parameters Array containing parameters to pass to function
    *       or method.  Pass in empty array to match tasks with no parameters.
    *       (OPTIONAL)
    * @param int $Priority Priority to assign to task.  (OPTIONAL)
    * @param string $Description Text description of task.  (OPTIONAL)
    * @return int Number of tasks queued that match supplied parameters.
    */
    public function GetQueuedTaskCount($Callback = NULL, $Parameters = NULL,
            $Priority = NULL, $Description = NULL)
    {
        $Query = "SELECT COUNT(*) AS TaskCount FROM TaskQueue";
        $Sep = " WHERE";
        if ($Callback !== NULL)
        {
            $Query .= $Sep." Callback = '".addslashes(serialize($Callback))."'";
            $Sep = " AND";
        }
        if ($Parameters !== NULL)
        {
            $Query .= $Sep." Parameters = '".addslashes(serialize($Parameters))."'";
            $Sep = " AND";
        }
        if ($Priority !== NULL)
        {
            $Query .= $Sep." Priority = ".intval($Priority);
            $Sep = " AND";
        }
        if ($Description !== NULL)
        {
            $Query .= $Sep." Description = '".addslashes($Description)."'";
        }
        return $this->DB->Query($Query, "TaskCount");
    }

    /**
    * Retrieve list of tasks currently running.
    * @param int $Count Number to retrieve.  (OPTIONAL, defaults to 100)
    * @param int $Offset Offset into queue to start retrieval.  (OPTIONAL)
    * @return Array with task IDs for index and task info for values.  Task info
    *       is stored as associative array with "Callback" and "Parameter" indices.
    */
    public function GetRunningTaskList($Count = 100, $Offset = 0)
    {
        return $this->GetTaskList("SELECT * FROM RunningTasks"
                ." WHERE StartedAt >= '".date("Y-m-d H:i:s",
                        (time() - $GLOBALS["AF"]->MaxExecutionTime()))."'"
                ." ORDER BY StartedAt", $Count, $Offset);
    }

    /**
    * Retrieve count of tasks currently running.
    * @return Number of running tasks.
    */
    public function GetRunningTaskCount()
    {
        return $this->DB->Query("SELECT COUNT(*) AS Count FROM RunningTasks"
                ." WHERE StartedAt >= '".date("Y-m-d H:i:s",
                        (time() - $GLOBALS["AF"]->MaxExecutionTime()))."'",
                "Count");
    }

    /**
    * Retrieve list of tasks currently orphaned.
    * @param int $Count Number to retrieve.  (OPTIONAL, defaults to 100)
    * @param int $Offset Offset into queue to start retrieval.  (OPTIONAL)
    * @return Array with task IDs for index and task info for values.  Task info
    *       is stored as associative array with "Callback" and "Parameter" indices.
    */
    public function GetOrphanedTaskList($Count = 100, $Offset = 0)
    {
        return $this->GetTaskList("SELECT * FROM RunningTasks"
                ." WHERE StartedAt < '".date("Y-m-d H:i:s",
                        (time() - $GLOBALS["AF"]->MaxExecutionTime()))."'"
                ." ORDER BY StartedAt", $Count, $Offset);
    }

    /**
    * Retrieve current number of orphaned tasks.
    * @return Number of orphaned tasks.
    */
    public function GetOrphanedTaskCount()
    {
        return $this->DB->Query("SELECT COUNT(*) AS Count FROM RunningTasks"
                ." WHERE StartedAt < '".date("Y-m-d H:i:s",
                        (time() - $GLOBALS["AF"]->MaxExecutionTime()))."'",
                "Count");
    }

    /**
    * Move orphaned task back into queue.
    * @param int $TaskId Task ID.
    * @param int $NewPriority New priority for task being requeued.  (OPTIONAL)
    */
    public function ReQueueOrphanedTask($TaskId, $NewPriority = NULL)
    {
        $this->DB->Query("LOCK TABLES TaskQueue WRITE, RunningTasks WRITE");
        $this->DB->Query("INSERT INTO TaskQueue"
                ." (Callback,Parameters,Priority,Description) "
                ."SELECT Callback, Parameters, Priority, Description"
                ." FROM RunningTasks WHERE TaskId = ".intval($TaskId));
        if ($NewPriority !== NULL)
        {
            $NewTaskId = $this->DB->LastInsertId();
            $this->DB->Query("UPDATE TaskQueue SET Priority = "
                            .intval($NewPriority)
                    ." WHERE TaskId = ".intval($NewTaskId));
        }
        $this->DB->Query("DELETE FROM RunningTasks WHERE TaskId = ".intval($TaskId));
        $this->DB->Query("UNLOCK TABLES");
    }

    /**
    * Set whether to requeue the currently-running background task when
    * it completes.
    * @param bool $NewValue If TRUE, current task will be requeued.  (OPTIONAL,
    *       defaults to TRUE)
    */
    public function RequeueCurrentTask($NewValue = TRUE)
    {
        $this->RequeueCurrentTask = $NewValue;
    }

    /**
    * Remove task from task queues.
    * @param int $TaskId Task ID.
    * @return int Number of tasks removed.
    */
    public function DeleteTask($TaskId)
    {
        $this->DB->Query("DELETE FROM TaskQueue WHERE TaskId = ".intval($TaskId));
        $TasksRemoved = $this->DB->NumRowsAffected();
        $this->DB->Query("DELETE FROM RunningTasks WHERE TaskId = ".intval($TaskId));
        $TasksRemoved += $this->DB->NumRowsAffected();
        return $TasksRemoved;
    }

    /**
    * Retrieve task info from queue (either running or queued tasks).
    * @param int $TaskId Task ID.
    * @return Array with task info for values or NULL if task is not found.
    *       Task info is stored as associative array with "Callback" and
    *       "Parameter" indices.
    */
    public function GetTask($TaskId)
    {
        # assume task will not be found
        $Task = NULL;

        # look for task in task queue
        $this->DB->Query("SELECT * FROM TaskQueue WHERE TaskId = ".intval($TaskId));

        # if task was not found in queue
        if (!$this->DB->NumRowsSelected())
        {
            # look for task in running task list
            $this->DB->Query("SELECT * FROM RunningTasks WHERE TaskId = "
                    .intval($TaskId));
        }

        # if task was found
        if ($this->DB->NumRowsSelected())
        {
            # if task was periodic
            $Row = $this->DB->FetchRow();
            if ($Row["Callback"] ==
                    serialize(array("ApplicationFramework", "RunPeriodicEvent")))
            {
                # unpack periodic task callback
                $WrappedCallback = unserialize($Row["Parameters"]);
                $Task["Callback"] = $WrappedCallback[1];
                $Task["Parameters"] = $WrappedCallback[2];
            }
            else
            {
                # unpack task callback and parameters
                $Task["Callback"] = unserialize($Row["Callback"]);
                $Task["Parameters"] = unserialize($Row["Parameters"]);
            }
        }

        # return task to caller
        return $Task;
    }

    /**
    * Get/set whether automatic task execution is enabled.  (This does not
    * prevent tasks from being manually executed.)
    * @param bool $NewValue TRUE to enable or FALSE to disable.  (OPTIONAL)
    * @param bool $Persistent If TRUE the new value will be saved (i.e.
    *       persistent across page loads), otherwise the value will apply to
    *       just the current page load.  (OPTIONAL, defaults to FALSE)
    * @return bool Returns TRUE if automatic task execution is enabled or
    *       otherwise FALSE.
    */
    public function TaskExecutionEnabled(
            $NewValue = DB_NOVALUE, $Persistent = FALSE)
    {
        return $this->UpdateSetting(__FUNCTION__, $NewValue, $Persistent);
    }

    /**
    * Get/set maximum number of tasks to have running simultaneously.
    * @param int $NewValue New setting for max number of tasks.  (OPTIONAL)
    * @param bool $Persistent If TRUE the new value will be saved (i.e.
    *       persistent across page loads), otherwise the value will apply to
    *       just the current page load.  (OPTIONAL, defaults to FALSE)
    * @return Current maximum number of tasks to run at once.
    */
    public function MaxTasks($NewValue = DB_NOVALUE, $Persistent = FALSE)
    {
        return $this->UpdateSetting("MaxTasksRunning", $NewValue, $Persistent);
    }

    /**
    * Get printable synopsis for task callback.  Any string values in the
    * callback parameter list will be escaped with htmlspecialchars().
    * @param array $TaskInfo Array of task info as returned by GetTask().
    * @return string Task callback synopsis string.
    * @see AFTaskManager::GetTask()
    */
    public static function GetTaskCallbackSynopsis(array $TaskInfo)
    {
        # if task callback is function use function name
        $Callback = $TaskInfo["Callback"];
        $Name = "";
        if (!is_array($Callback))
        {
            $Name = $Callback;
        }
        else
        {
            # if task callback is object
            if (is_object($Callback[0]))
            {
                # if task callback is encapsulated ask encapsulation for name
                if (method_exists($Callback[0], "GetCallbackAsText"))
                {
                    $Name = $Callback[0]->GetCallbackAsText();
                }
                # else assemble name from object
                else
                {
                    $Name = get_class($Callback[0]) . "::" . $Callback[1];
                }
            }
            # else assemble name from supplied info
            else
            {
                $Name= $Callback[0] . "::" . $Callback[1];
            }
        }

        # if parameter array was supplied
        $Parameters = $TaskInfo["Parameters"];
        $ParameterString = "";
        if (is_array($Parameters))
        {
            # assemble parameter string
            $Separator = "";
            foreach ($Parameters as $Parameter)
            {
                $ParameterString .= $Separator;
                if (is_int($Parameter) || is_float($Parameter))
                {
                    $ParameterString .= $Parameter;
                }
                else if (is_string($Parameter))
                {
                    $ParameterString .= "\"".htmlspecialchars($Parameter)."\"";
                }
                else if (is_array($Parameter))
                {
                    $ParameterString .= "ARRAY";
                }
                else if (is_object($Parameter))
                {
                    $ParameterString .= "OBJECT";
                }
                else if (is_null($Parameter))
                {
                    $ParameterString .= "NULL";
                }
                else if (is_bool($Parameter))
                {
                    $ParameterString .= $Parameter ? "TRUE" : "FALSE";
                }
                else if (is_resource($Parameter))
                {
                    $ParameterString .= get_resource_type($Parameter);
                }
                else
                {
                    $ParameterString .= "????";
                }
                $Separator = ", ";
            }
        }

        # assemble name and parameters and return result to caller
        return $Name."(".$ParameterString.")";
    }

    /**
    * Determine current priority if running in background.
    * @return int Current background priority (PRIORITY_ value), or NULL
    *       if not currently running in background.
    */
    public function GetCurrentBackgroundPriority()
    {
        return isset($this->RunningTask)
                ? $this->RunningTask["Priority"] : NULL;
    }

    /**
    * Get next higher possible background task priority.  If already at the
    * highest priority, the same value is returned.
    * @param int $Priority Background priority (PRIORITY_ value).  (OPTIONAL,
    *       defaults to current priority if running in background, or NULL if
    *       running in foreground)
    * @return integer|null Next higher background priority, or NULL if no priority
    *       specified and currently running in foreground.
    */
    public function GetNextHigherBackgroundPriority($Priority = NULL)
    {
        if ($Priority === NULL)
        {
            $Priority = $this->GetCurrentBackgroundPriority();
            if ($Priority === NULL)
            {
                return NULL;
            }
        }
        return ($Priority > self::PRIORITY_HIGH)
                ? ($Priority - 1) : self::PRIORITY_HIGH;
    }

    /**
    * Get next lower possible background task priority.  If already at the
    * lowest priority, the same value is returned.
    * @param int $Priority Background priority (PRIORITY_ value).  (OPTIONAL,
    *       defaults to current priority if running in background, or NULL if
    *       running in foreground)
    * @return integer|null Next lower background priority, or NULL if no priority
    *       specified and currently running in foreground.
    */
    public function GetNextLowerBackgroundPriority($Priority = NULL)
    {
        if ($Priority === NULL)
        {
            $Priority = $this->GetCurrentBackgroundPriority();
            if ($Priority === NULL)
            {
                return NULL;
            }
        }
        return ($Priority < self::PRIORITY_BACKGROUND)
                ? ($Priority + 1) : self::PRIORITY_BACKGROUND;
    }

    /**
    * Run any queued background tasks until either remaining PHP execution
    * time or available memory run too low.
    */
    public function RunQueuedTasks()
    {
        # if there are tasks in the queue
        if ($this->GetTaskQueueSize())
        {
            # tell PHP to garbage collect to give as much memory as possible for tasks
            gc_collect_cycles();

            # turn on output buffering to (hopefully) record any crash output
            ob_start();

            # lock tables to prevent anyone else from running a task
            $this->DB->Query("LOCK TABLES TaskQueue WRITE, RunningTasks WRITE");

            # while there is time and memory left
            #       and a task to run
            #       and an open slot to run it in
            $MinimumTimeToRunAnotherTask = 65;
            while (($GLOBALS["AF"]->GetSecondsBeforeTimeout()
                            > $MinimumTimeToRunAnotherTask)
                    && (ApplicationFramework::GetPercentFreeMemory()
                            > $this->BackgroundTaskMinFreeMemPercent)
                    && $this->GetTaskQueueSize()
                    && ($this->GetRunningTaskCount() < $this->MaxTasks()))
            {
                # look for task at head of queue
                $this->DB->Query("SELECT * FROM TaskQueue"
                        ." ORDER BY Priority, TaskId LIMIT 1");
                $Task = $this->DB->FetchRow();

                # move task from queued list to running tasks list
                $this->DB->Query("INSERT INTO RunningTasks "
                        ."(TaskId,Callback,Parameters,Priority,Description) "
                        ."SELECT * FROM TaskQueue WHERE TaskId = "
                        .intval($Task["TaskId"]));
                $this->DB->Query("DELETE FROM TaskQueue WHERE TaskId = "
                        .intval($Task["TaskId"]));

                # release table locks to again allow other sessions to run tasks
                $this->DB->Query("UNLOCK TABLES");

                # update the "last run" time
                $this->DB->Query("UPDATE ApplicationFrameworkSettings"
                        ." SET LastTaskRunAt = '".date("Y-m-d H:i:s")."'");

                # run task
                $this->RunTask($Task);

                # lock tables to prevent anyone else from running a task
                $this->DB->Query("LOCK TABLES TaskQueue WRITE, RunningTasks WRITE");
            }

            $this->ResetTaskIdGeneratorIfNecessary();

            # make sure tables are released
            $this->DB->Query("UNLOCK TABLES");
        }
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $BackgroundTaskMemLeakLogThreshold = 10;    # percentage of max mem
    private $BackgroundTaskMinFreeMemPercent = 25;
    private $DB;
    private $MaxRunningTasksToTrack = 250;
    private $RequeueCurrentTask;
    private $RunningTask;
    private $Settings;

    /**
    * Set to TRUE to not close browser connection before running
    *       background tasks (useful when debugging)
    */
    private $NoTSR = FALSE;

    /**
    * Load our settings from database, initializing them if needed.
    * @throws Exception If unable to load settings.
    */
    private function LoadSettings()
    {
        # read settings in from database
        $this->DB->Query("SELECT * FROM ApplicationFrameworkSettings");
        $this->Settings = $this->DB->FetchRow();

        # if settings were not previously initialized
        if ($this->Settings === FALSE)
        {
            # initialize settings in database
            $this->DB->Query("INSERT INTO ApplicationFrameworkSettings"
                    ." (LastTaskRunAt) VALUES ('2000-01-02 03:04:05')");

            # read new settings in from database
            $this->DB->Query("SELECT * FROM ApplicationFrameworkSettings");
            $this->Settings = $this->DB->FetchRow();

            # bail out if reloading new settings failed
            if ($this->Settings === FALSE)
            {
                throw new Exception(
                        "Unable to load application framework settings.");
            }
        }
    }

    /**
    * Retrieve list of tasks with specified query.
    * @param string $DBQuery Database query.
    * @param int $Count Number to retrieve.
    * @param int $Offset Offset into queue to start retrieval.
    * @return array Array with task IDs for index and task info for values.
    *       Task info is stored as associative array with "Callback" and
    *       "Parameter" indices.
    */
    private function GetTaskList($DBQuery, $Count, $Offset)
    {
        $this->DB->Query($DBQuery." LIMIT ".intval($Offset).",".intval($Count));
        $Tasks = array();
        while ($Row = $this->DB->FetchRow())
        {
            $Tasks[$Row["TaskId"]] = $Row;
            if ($Row["Callback"] ==
                    serialize(array("ApplicationFramework", "RunPeriodicEvent")))
            {
                $WrappedCallback = unserialize($Row["Parameters"]);
                $Tasks[$Row["TaskId"]]["Callback"] = $WrappedCallback[1];
                $Tasks[$Row["TaskId"]]["Parameters"] = NULL;
            }
            else
            {
                $Tasks[$Row["TaskId"]]["Callback"] = unserialize($Row["Callback"]);
                $Tasks[$Row["TaskId"]]["Parameters"] = unserialize($Row["Parameters"]);
            }
        }
        return $Tasks;
    }

    /**
    * Run given task.
    * @param array $Task Array of task info.
    */
    private function RunTask($Task)
    {
        # unpack stored task info
        $TaskId = $Task["TaskId"];
        $Callback = unserialize($Task["Callback"]);
        $Parameters = unserialize($Task["Parameters"]);

        # attempt to load task callback if not already available
        $GLOBALS["AF"]->LoadFunction($Callback);

        # clear task requeue flag
        $this->RequeueCurrentTask = FALSE;

        # save amount of free memory for later comparison
        $BeforeFreeMem = ApplicationFramework::GetFreeMemory();

        # run task
        $this->RunningTask = $Task;
        if ($Parameters)
        {
            call_user_func_array($Callback, $Parameters);
        }
        else
        {
            call_user_func($Callback);
        }
        unset($this->RunningTask);

        # log if task leaked significant memory
        $this->LogTaskMemoryLeakIfAny($TaskId, $BeforeFreeMem);

        # if task requeue requested
        if ($this->RequeueCurrentTask)
        {
            # move task from running tasks list to queue
            $this->RequeueRunningTask($TaskId);
        }
        else
        {
            # remove task from running tasks list
            $this->DB->Query("DELETE FROM RunningTasks"
                    ." WHERE TaskId = ".intval($TaskId));
        }

        # prune running tasks list if necessary
        $RunningTasksCount = $this->DB->Query(
                "SELECT COUNT(*) AS TaskCount FROM RunningTasks", "TaskCount");
        if ($RunningTasksCount > $this->MaxRunningTasksToTrack)
        {
            $this->DB->Query("DELETE FROM RunningTasks ORDER BY StartedAt"
                    ." LIMIT ".($RunningTasksCount - $this->MaxRunningTasksToTrack));
        }
    }

    /**
    * Requeue running task, moving it from the running tasks list (in the DB)
    * to the queued tasks list.
    * @param int $TaskId ID of running task.
    */
    private function RequeueRunningTask($TaskId)
    {
        $this->DB->Query("LOCK TABLES TaskQueue WRITE, RunningTasks WRITE");
        $this->DB->Query("INSERT INTO TaskQueue"
                ." (Callback,Parameters,Priority,Description)"
                ." SELECT Callback,Parameters,Priority,Description"
                ." FROM RunningTasks WHERE TaskId = ".intval($TaskId));
        $this->DB->Query("DELETE FROM RunningTasks WHERE TaskId = ".intval($TaskId));
        $this->DB->Query("UNLOCK TABLES");
    }

    /**
    * If TaskIds are nearing their max value, TRUNCATE the TaskQueue
    * table to reset them. Necessary because MySQL will refuse to
    * INSERT new rows after an AUTO_INCREMENT id hits its max value.
    */
    private function ResetTaskIdGeneratorIfNecessary()
    {
        $this->DB->Query("LOCK TABLES TaskQueue WRITE");
        if (($GLOBALS["AF"]->GetSecondsBeforeTimeout() > 30)
                && ($this->GetTaskQueueSize() == 0)
                && ($this->DB->GetNextInsertId("TaskQueue")
                        > (Database::INT_MAX_VALUE * 0.90)))
        {
            $this->DB->Query("TRUNCATE TABLE TaskQueue");
        }
        $this->DB->Query("UNLOCK TABLES");
    }

    /**
    * Log memory leak (if any) when task was executed.
    * @param int $TaskId ID of task.
    * @param int $StartingFreeMemory Amount of free memory in bytes before
    *       task was run.
    */
    private function LogTaskMemoryLeakIfAny($TaskId, $StartingFreeMemory)
    {
        # tell PHP to garbage collect to free up any memory no longer used
        gc_collect_cycles();

        # calculate the logging threshold
        $LeakThreshold = ApplicationFramework::GetPhpMemoryLimit()
                * ($this->BackgroundTaskMemLeakLogThreshold / 100);

        # calculate the amount of memory used by task
        $EndingFreeMemory = ApplicationFramework::GetFreeMemory();
        $MemoryUsed = $StartingFreeMemory - $EndingFreeMemory;

        # if amount of memory used is over threshold
        if ($MemoryUsed > $LeakThreshold)
        {
            # log memory leak
            $TaskSynopsis = self::GetTaskCallbackSynopsis($this->GetTask($TaskId));
            $GLOBALS["AF"]->LogError(ApplicationFramework::LOGLVL_DEBUG,
                    "Task ".$TaskSynopsis." leaked "
                            .number_format($MemoryUsed)." bytes.");
        }
    }

    /**
    * Convenience function for getting/setting our settings.
    * @param string $FieldName Name of database field used to store setting.
    * @param mixed $NewValue New value for setting.  (OPTIONAL)
    * @param bool $Persistent If TRUE the new value will be saved (i.e.
    *       persistent across page loads), otherwise the value will apply to
    *       just the current page load.  (OPTIONAL, defaults to TRUE)
    * @return mixed Current value for setting.
    */
    private function UpdateSetting(
            $FieldName, $NewValue = DB_NOVALUE, $Persistent = TRUE)
    {
        static $LocalSettings;
        if ($NewValue !== DB_NOVALUE)
        {
            if ($Persistent)
            {
                $LocalSettings[$FieldName] = $this->DB->UpdateValue(
                        "ApplicationFrameworkSettings",
                        $FieldName, $NewValue, NULL, $this->Settings);
            }
            else
            {
                $LocalSettings[$FieldName] = $NewValue;
            }
        }
        elseif (!isset($LocalSettings[$FieldName]))
        {
            $LocalSettings[$FieldName] = $this->DB->UpdateValue(
                    "ApplicationFrameworkSettings",
                    $FieldName, $NewValue, NULL, $this->Settings);
        }
        return $LocalSettings[$FieldName];
    }
}
