<?PHP
#
#   FILE:  API.php (DrupalSync plugin)
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2012-2016 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis
#

/**
* Process a rest command.
* @param array $Message REST API message to process.
* @return Status of command.
*/
function ProcessRestCommand($Message)
{
    $UserFactory = new CWUserFactory();

    # load user from provided email when appropriate based on command
    switch ($Message["Data"]["Cmd"])
    {
        case "UserLogin":
        case "UserLogout":
        case "DeleteUser":
        case "UpdateEmail":
        case "UpdatePassword":
            $Matches = $UserFactory->GetMatchingUsers(
                preg_quote($Message["Data"]["UserEmail"]), "EMail");

            if (count($Matches)==0)
            {
                return array(
                    "Status" => "Error",
                    "Message" => "User not found.");
            }
            elseif (count($Matches)>1)
            {
                return array(
                    "Status" => "Error",
                    "Message" => "Multiple users match this email.");
            }
            else
            {
                $User = new CWUser(key($Matches));
            }
    }

    switch ($Message["Data"]["Cmd"])
    {
        case "UserLogin":
            $Result = $User->Login(
                $User->Name(), $Message["Data"]["UserPassword"]);

            if ($Result != U_OKAY)
            {
                return array(
                    "Status" => "Error",
                    "Message" => "Login failed.",
                    "UserErrorCode" => $Result);
            }

            $Reply = array(
                "Status" => "OK",
                "Message" => "Login successful.",
                "Cookies" => array(
                    session_name() => session_id(),
                ),
            );

            $_SESSION["DrupalSync_NeedsLogout"] = $Message["Data"]["Cookies"];

            return $Reply;
            break;

        case "UserLogout":
            if (session_status() == PHP_SESSION_ACTIVE)
            {
                session_abort();
            }

            session_id(
                $Message["Data"]["Cookies"][session_name()]);
            session_start();

            $User->Logout();
            return array(
                "Status" => "OK",
                "Message" => "User successfully logged out.");
            break;

        case "CreateUser":
            # see if a user with this email already exists
            $Matches = $UserFactory->GetMatchingUsers(
                preg_quote($Message["Data"]["UserEmail"]),
                "EMail");

            # if so, report an error
            if (count($Matches)>0)
            {
                return array(
                    "Status" => "Error",
                    "Message" => "User already exists.");
            }

            # if not, we'll need to create the user

            $TgtUserName = $Message["Data"]["UserName"];
            if ($UserFactory->UserNameExists($TgtUserName))
            {
                $TgtUserName = CWUserFactory::GenerateUniqueUsernameFromEmail(
                    $Message["Data"]["UserEmail"]);
            }

            $TmpUser = $UserFactory->CreateNewUser(
                $TgtUserName,
                $Message["Data"]["Password"],
                $Message["Data"]["Password"],
                $Message["Data"]["UserEmail"],
                $Message["Data"]["UserEmail"],
                array(U_ILLEGALPASSWORD, U_ILLEGALPASSWORDAGAIN,
                      U_ILLEGALEMAIL, U_ILLEGALEMAILAGAIN));

            if (!is_object($TmpUser) || $TmpUser->Status() != U_OKAY)
            {
                $Reply = array(
                    "Status" => "Error",
                    "Message" => "Could not create user.",
                    "GotUserObject" => is_object($TmpUser));

                if (is_object($TmpUser))
                {
                    $Reply["UserErrorCode"] = $TmpUser->Status();
                }
                return $Reply;
            }

            return array(
                "Status" => "OK",
                "Message" => "User successfully created.",
                "UserName" => $TgtUserName);
            break;

        case "DeleteUser":
            $User->Delete();

            return array(
                "Status" => "OK",
                "Message" => "User successfully deleted.",
            );
            break;

        case "UpdatePassword":
            $StoredPassword = $User->Get("UserPassword");

            $HashedOldPassword = crypt(
                $Message["Data"]["OldPassword"], $StoredPassword);

            # if password was good
            if ($StoredPassword != $HashedNewPassword)
            {
                return array(
                    "Status" => "Error",
                    "Message" => "Unable to change password, bad current password.");
            }

            # change password (skipping CWIS password rules)
            $User->SetPassword(
                $Message["Data"]["NewPassword"]);

            return array(
                "Status" => "OK",
                "Message" => "Password successfully changed.");
            break;

        case "UpdateEmail":
            $TgtMatches = $UserFactory->GetMatchingUsers(
                preg_quote($Message["Data"]["NewUserEmail"]),
                "EMail");

            if (count($TgtMatches)>0)
            {
                return array(
                    "Status" => "Error",
                    "Message" => "Another user already exists with this email address.");
            }

            $User->Set("EMail",
                       $Message["Data"]["NewUserEmail"]);
            return array(
                "Status" => "Error",
                "Message" => "Email successfully changed.");
            break;

        case "GetUserInfo":
            # verify that specified user exists, returning an error if not
            $UFactory = new CWUserFactory();
            if (!$UFactory->UserNameExists($Message["Data"]["UserName"]))
            {
                return array(
                    "Status" => "Error",
                    "Message" => "User not found.");
            }

            # try to create a CWUser, returning an error if this fails
            try
            {
                $User = new CWUser($Message["Data"]["UserName"]);
            }
            catch (Exception $e)
            {
                return array(
                    "Status" => "Error",
                    "Message" => "Exception thrown trying to create CWUser.");
            }

            # return an error if the CWUser's status is not U_OKAY
            if ($User->Status() != U_OKAY)
            {
                return array(
                    "Status" => "Error",
                    "Message" => "CWUser created, but status is not U_OKAY.");
            }

            $StoredPassword = $User->Get("UserPassword");
            $HashedPassword = crypt($Message["Data"]["UserPassword"],
                                    $StoredPassword);

            if ($StoredPassword != $HashedPassword)
            {
                return array(
                    "Status" => "Error",
                    "Message" => "Incorrect password.");
            }

            return array(
                "Status" => "OK",
                "UserEmail" => $User->Get("EMail"),
            );
            break;

        default:
            return array(
                "Status" => "Error",
                "Message" => "Command not understood.");
            break;
    }
}


$GLOBALS["AF"]->SuppressHTMLOutput();

$MyPlugin = $GLOBALS["G_PluginManager"]->GetPluginForCurrentPage();

$Message = $MyPlugin->RESTHelper()->DecodeEncryptedMessage($_POST);

if ($Message["Status"] != "OK")
{
    $Reply = $Message;
}
else
{
    $Reply = ProcessRestCommand($Message);
}

print json_encode(
    $MyPlugin->RESTHelper()->EncodeEncryptedMessage(
        $Reply));