<?PHP

class Resource_Test extends PHPUnit\Framework\TestCase
{
    /**
    * Prior to running any of the tests, this function is
    * run. It creates all of the test Metadata fields and adds
    * them to class variables $TestFieldIds and $TestFields
    * so each function may use them.
    */
    public static function setUpBeforeClass()
    {
        # construct the schema object
        $Schema = new MetadataSchema(MetadataSchema::SCHEMAID_DEFAULT);

        self::$TestFieldIds = array();

        # outline fields to be created
        self::$TestFields = array(
          "Test Text Field" => MetadataSchema::MDFTYPE_TEXT,
          "Test Timestamp Field" => MetadataSchema::MDFTYPE_TIMESTAMP,
          "Test Paragraph Field" => MetadataSchema::MDFTYPE_PARAGRAPH,
          "Test Url Field" => MetadataSchema::MDFTYPE_URL,
          "Test Reference Field" => MetadataSchema::MDFTYPE_REFERENCE,
          "Test User Field" => MetadataSchema::MDFTYPE_USER,
          "Test Option Field" => MetadataSchema::MDFTYPE_OPTION,
          "Test CName Field" => MetadataSchema::MDFTYPE_CONTROLLEDNAME,
          "Test Tree Field" => MetadataSchema::MDFTYPE_TREE,
          "Test Date Field" => MetadataSchema::MDFTYPE_DATE,
          "Test Flag Field" => MetadataSchema::MDFTYPE_FLAG,
          "Test Number Field" => MetadataSchema::MDFTYPE_NUMBER,
          "Test Point Field" => MetadataSchema::MDFTYPE_POINT,
        );

        # create the fields
        foreach (self::$TestFields as $FieldName => $FieldType)
        {
            $TmpField = $Schema->GetItemByName($FieldName);
            if ($TmpField === NULL)
            {
                $TmpField = $Schema->AddField($FieldName, $FieldType);
            }
            $TmpField->IsTempItem(FALSE);
            self::$TestFieldIds[$FieldName] = $TmpField->Id();
        }

        # Resource::Create() expects a user to be logged in,
        # so log in an admin user
        $UFactory = new CWUserFactory();
        $Users = $UFactory->GetUsersWithPrivileges(
                PRIV_RESOURCEADMIN, PRIV_COLLECTIONADMIN);
        $UserIds = array_keys($Users);
        $AdminUserId = array_pop($UserIds);
        self::$AdminUser = new CWUser($AdminUserId);
        $GLOBALS["G_User"]->Login(self::$AdminUser->Name(), "", TRUE);

        # Create Classification, ControlledName, and Option values
        self::$TestClassification = Classification::Create(
            "Test Classification", self::$TestFieldIds['Test Tree Field']);
        self::$TestControlledName = ControlledName::Create(
            "Test Controlled Name", self::$TestFieldIds['Test CName Field']);
        self::$TestOptionCName = ControlledName::Create(
            "Test Option Name", self::$TestFieldIds['Test Option Field']);
    }

    /**
    * After to running the tests, this function is
    * run. It deletes all of the test Metadata fields.
    */
    public static function tearDownAfterClass()
    {
        # construct the schema object
        $Schema = new MetadataSchema();
        $Database = new Database();

        # drop all of the test fields
        foreach (self::$TestFieldIds as $FieldName => $FieldId)
        {
            $Schema->DropField($FieldId);
        }

        self::$TestClassification->Delete();
        self::$TestControlledName->Delete();
        self::$TestOptionCName->Delete();
    }

    /**
    * This function exercises the Resource get and set methods for
    * each Metadata types using the fields created in setUpBeforeClass().
    */
    public function testResource()
    {
        # create test-specific objects
        $TestResource = Resource::Create(MetadataSchema::SCHEMAID_DEFAULT);
        $this->assertTrue($TestResource->IsTempResource(),
            "Check that newly created resources are temporary.");

        $this->assertFalse($TestResource->IsTempResource(FALSE),
            "Check resources can be set permanent.");

        # test Comments features
        $this->CheckComments($TestResource);

        # test Ratings features
        $this->CheckRatings($TestResource);

        # test permissions-related functions
        $this->CheckPermissions($TestResource);

        # test get, set, and clear
        $TestReferenceResource = Resource::Create(MetadataSchema::SCHEMAID_DEFAULT);
        $TestReferenceResource->IsTempResource(FALSE);
        $this->CheckGetSetClear($TestResource, $TestReferenceResource);

        # check that resource schemas can be retrieved
        $this->CheckGetSchemaForResource(
            $TestResource, $TestReferenceResource);

        # check that GetAsArray works
        $this->CheckGetAsArray(
            $TestResource, $TestReferenceResource);

        # check that perm resource can be made temporary and don't
        # lose any values in the process
        $this->CheckTempToggle($TestResource);


        # clean up function-specific objects
        $TestResource->Delete();
        $TestReferenceResource->Delete();
    }

    /**
    * Check that get, set, and clear all function for all tested field types.
    * @param Resource $Resource Resource to test.
    * @param Resource $RefResource Resource to use as a reference field value.
    */
    private function CheckGetSetClear($Resource, $RefResource)
    {
        # test get, set, and clear for each test field
        foreach (self::$TestFieldIds as $FieldName => $FieldId)
        {
            $Field = new MetadataField($FieldId);

            # whether, before testing equivalence, we need to pop the
            # returned value out of an array
            $BeforeTestArrayShift = FALSE;

            # if we're testing the object return, this is the object we'll compare it to.
            unset($TestObject);

            switch ($Field->Type())
            {
                case MetadataSchema::MDFTYPE_TEXT:
                    $TgtVal = "A test title";
                    break;

                case MetadataSchema::MDFTYPE_URL:
                    $TgtVal = "http://testtesttest.com";
                    break;

                case MetadataSchema::MDFTYPE_PARAGRAPH:
                    $TgtVal = "I am a test paragraph.";
                    break;

                case MetadataSchema::MDFTYPE_NUMBER:
                    $TgtVal = "0";
                    break;

                case MetadataSchema::MDFTYPE_FLAG:
                    $TgtVal = "1";
                    break;

                case MetadataSchema::MDFTYPE_DATE:
                    $TgtVal = date("Y-m-d");
                    $TestObject = new Date(strval($TgtVal));
                    $TestObjectType = 'Date';
                    $TestFunctionName = 'BeginDate';
                    $TestFunctionArguments = NULL;
                    break;

                case MetadataSchema::MDFTYPE_TIMESTAMP:
                    $TgtVal = date("Y-m-d H:i:s");
                    break;

                case MetadataSchema::MDFTYPE_TREE:
                    $TgtVal = array();
                    $TgtVal[self::$TestClassification->Id()] = "Test Classification";
                    $TestObject = self::$TestClassification;
                    $TestObjectType = 'Classification';
                    $TestFunctionName = 'FullName';
                    $TestFunctionArguments = NULL;
                    $BeforeTestArrayShift = TRUE;
                    break;

                case MetadataSchema::MDFTYPE_CONTROLLEDNAME:
                    $TgtVal = array();
                    $TgtVal[self::$TestControlledName->Id()] = "Test Controlled Name";
                    $TestObject = self::$TestControlledName;
                    $TestObjectType = 'ControlledName';
                    $TestFunctionName = 'Name';
                    $TestFunctionArguments = NULL;
                    $BeforeTestArrayShift = TRUE;
                    break;

                case MetadataSchema::MDFTYPE_OPTION:
                    $TgtVal = array();
                    $TgtVal[self::$TestOptionCName->Id()] = "Test Option Name";
                    $TestObject = self::$TestOptionCName;
                    $TestObjectType = 'ControlledName';
                    $TestFunctionName = 'Name';
                    $TestFunctionArguments = NULL;
                    $BeforeTestArrayShift = TRUE;
                    break;

                case MetadataSchema::MDFTYPE_USER:
                    $UserId = $GLOBALS["G_User"]->Id();
                    $TestObject = new CWUser($UserId);
                    $TgtVal = array( $UserId => $TestObject->Name() );
                    $TestObjectType = 'CWUser';
                    $TestFunctionName = 'Id';
                    $TestFunctionArguments = NULL;
                    $BeforeTestArrayShift = TRUE;
                    break;

                case MetadataSchema::MDFTYPE_POINT:
                    $TgtVal = array();
                    $TgtVal['X'] = 5;
                    $TgtVal['Y'] = 7;
                    break;

                case MetadataSchema::MDFTYPE_REFERENCE:
                    $TestObject = $RefResource;
                    $TgtVal = array();
                    $TgtVal[$RefResource->Id()] = $RefResource->Id();
                    $TestFunctionName = 'Id';
                    $TestObjectType = 'Resource';
                    $TestFunctionArguments = NULL;
                    $BeforeTestArrayShift = TRUE;
                    break;

                default:
                    throw new Exception("Data type not handled.");
                    break;

            }

            # set the value on the test resource
            $Resource->Set($Field, $TgtVal);

            # assert the default get returns the expected value
            $FieldTypeName = StdLib::GetConstantName(
                    "MetadataSchema", $Field->Type(), "MDFTYPE_");
            $this->assertEquals($TgtVal, $Resource->Get($Field),
                    "Check that value returned by Get() matches for field type "
                            .$FieldTypeName);

            $this->assertTrue($Resource->FieldIsSet($Field),
                    "Check that FieldIsSet() returns TRUE after setting value for field type "
                        .$FieldTypeName);

            $RCopy = new Resource($Resource->Id());
            $this->assertEquals($TgtVal, $RCopy->Get($Field),
                    "Check that value returned by Get() matches for field type w/ new resource"
                    .$FieldTypeName);

            if (isset($TestObject))
            {
                $ReturnedObject = $Resource->Get($Field, TRUE);

                if ($BeforeTestArrayShift)
                {
                    $ReturnedObject = array_shift($ReturnedObject);
                }

                $array_for_test_object = array($TestObject, $TestFunctionName);
                $array_for_returned_object = array($ReturnedObject, $TestFunctionName);

                if ($TestFunctionArguments !== NULL)
                {
                    $this->assertEquals(call_user_func(
                            $array_for_returned_object, $TestFunctionArguments),
                            call_user_func($array_for_test_object, $TestFunctionArguments));
                }
                else
                {
                    $this->assertEquals(call_user_func($array_for_returned_object),
                            call_user_func($array_for_test_object));
                }

                $this->assertInstanceOf($TestObjectType, $ReturnedObject);
            }

            # clear the value from the field
            $Resource->Clear($Field);

            switch ($Field->Type())
            {
                case MetadataSchema::MDFTYPE_TEXT:
                case MetadataSchema::MDFTYPE_URL:
                case MetadataSchema::MDFTYPE_PARAGRAPH:
                case MetadataSchema::MDFTYPE_DATE:
                case MetadataSchema::MDFTYPE_TIMESTAMP:
                case MetadataSchema::MDFTYPE_NUMBER:
                case MetadataSchema::MDFTYPE_FLAG:
                    $TgtVal = NULL;
                    break;

                case MetadataSchema::MDFTYPE_TREE:
                case MetadataSchema::MDFTYPE_CONTROLLEDNAME:
                case MetadataSchema::MDFTYPE_OPTION:
                case MetadataSchema::MDFTYPE_USER:
                case MetadataSchema::MDFTYPE_REFERENCE:
                    $TgtVal = array();
                    break;

                case MetadataSchema::MDFTYPE_POINT:
                    $TgtVal = array(
                        "X" => NULL,
                        "Y" => NULL );
                    break;

                default:
                    throw new Exception("Data type not handled.");
                    break;

            }

            $this->assertEquals($TgtVal, $Resource->Get($Field));

            $this->assertFalse($Resource->FieldIsSet($Field),
                    "Check that FieldIsSet() returns FALSE after clearing value for field type "
                        .$FieldTypeName);

            $RCopy = new Resource($Resource->Id());
            $this->assertEquals($TgtVal, $RCopy->Get($Field),
                    "Check that value returned by Get() matches for field type w/ new resource"
                    .$FieldTypeName);
        }
    }

    /**
    * Check that newly created resources have no comments, that a
    * comment can be added, and that this comment can be removed.
    * @param Resource $Resource Newly-created Resource to test.
    */
    private function CheckComments($Resource)
    {
        $this->assertEquals(0, $Resource->NumberOfComments(),
            "Check that newly created resources have no comments.");
        $this->assertNull($Resource->Comments(),
            "Check that newly created resources have null comment list.");

        $TestComment = Message::Create();
        $TestComment->ParentType(Message::PARENTTYPE_RESOURCE);
        $TestComment->ParentId($Resource->Id());

        # reload resource to nuke internal caches
        $Resource = new Resource($Resource->Id());

        $this->assertEquals(1, $Resource->NumberOfComments(),
            "Check that NumberOfComments() is one after adding a single comment.");

        $RComments = $Resource->Comments();

        $this->assertTrue(is_array($RComments),
            "Check that Comments() returns an array.");

        $this->assertEquals(1, count($RComments),
            "Check that Comments() returns an array of length 1");

        $RComment = array_Shift($RComments);

        $this->assertTrue($RComment instanceof Message,
            "Check that the comment is a Message.");

        $this->assertEquals($TestComment->Id(), $RComment->Id(),
             "Check that the CommentId of the single Message in the array returned "
             ."by Comments() matches the Id of the test comment "
             ."that we just associated with the resource.");

        $TestComment->Destroy();

        # reload resource to nuke internal caches
        $Resource = new Resource($Resource->Id());

        $this->assertEquals(0, $Resource->NumberOfComments(),
            "Check that resource has no comments after deleting comment.");
        $this->assertNull($Resource->Comments(),
            "Check that resource has null comment list after deleting comment.");
    }

    /**
    * Check that newly created resources have no initial ratings, but
    * that they can be rated, and that this rating can be changed.
    * @param Resource $Resource Newly-created resource to test.
    */
    private function CheckRatings($Resource)
    {
        $this->assertEquals(0, $Resource->NumberOfRatings(),
            "Check that newly created resources have no ratings.");
        $this->assertEquals(0, $Resource->CumulativeRating(),
            "Check that newly created resources have no cumulative rating.");
        $this->assertEquals(0, $Resource->ScaledCumulativeRating(),
            "Check that newly created resources have no scaled cumulative rating.");

        # ratings checks
        $this->assertNull($Resource->Rating(),
            "Check that admin user hasn't rated this resource.");
        $this->assertEquals(25, $Resource->Rating(25),
            "Check that admin user can rate this resource.");
        $this->assertEquals(25, $Resource->Rating(),
            "Check that admin's rating was saved");
        $this->assertEquals(1, $Resource->NumberOfRatings(),
            "Check that number of ratings is correct.");
        $this->assertEquals(25, $Resource->CumulativeRating(),
            "Check that cumulative rating is correct.");
        $this->assertEquals(3, $Resource->ScaledCumulativeRating(),
            "Check that scaled cumulative rating is correct.");
        $this->assertEquals(50, $Resource->Rating(50),
            "Check that admin can change rating.");
        $this->assertEquals(1, $Resource->NumberOfRatings(),
            "Check that number of ratings is correct.");
        $this->assertEquals(50, $Resource->CumulativeRating(),
            "Check that cumulative rating is correct.");
        $this->assertEquals(5, $Resource->ScaledCumulativeRating(),
            "Check that scaled cumulative rating is correct.");

        $GLOBALS["G_User"]->Logout();
        $this->assertNull(
            $Resource->Rating(),
            "Check that anon user hasn't rated this resource.");
        $GLOBALS["G_User"]->Login(self::$AdminUser->Name(), "", TRUE);
    }

    /**
    * Check permissions functions --
    * UserCan(View|Edit|Author|Modify)(Field)?.  Assumes a schema
    * where newly created resources cannot be accessed at all by anon
    * users, but the administrative user in self::$AdminUser (having
    * PRIV_RESOURCEADMIN and PRIV_COLLECTIONADMIN) has full access --
    * this is the case for the Resource Schema in a default CWIS
    * install and on all Scout sites.
    * @param Resource $Resource Resource to test.
    */
    private function CheckPermissions($Resource)
    {
        $TitleField = $Resource->Schema()->GetFieldByMappedName("Title");
        foreach (["View", "Edit", "Author", "Modify"] as $Action)
        {
            $CheckFn = "UserCan".$Action;
            $FieldCheckFn = "UserCan".$Action."Field";

            $this->assertFalse(
                $Resource->$CheckFn(CWUser::GetAnonymousUser()),
                "Check that Anon users cannot ".strtolower($Action)
                ." a new Resource.");
            $this->assertFalse(
                $Resource->$FieldCheckFn(CWUser::GetAnonymousUser(), $TitleField),
                "Check that Anon users cannot ".strtolower($Action)
                ." the Title field on a new Resource.");

            $this->assertTrue(
                $Resource->UserCanView(self::$AdminUser),
                "Check that admin users can ".strtolower($Action)
                ." a new Resource.");
            $this->assertTrue(
                $Resource->$FieldCheckFn(self::$AdminUser, $TitleField),
                "Check that admin users can ".strtolower($Action)
                ." the Title field on a new Resource.");
        }

        $this->assertFalse(
            $Resource->UserCanViewMappedField(CWUser::GetAnonymousUser(), "Title"),
            "Check that Anon users cannot view mapped Title on a new Resource.");

        $this->assertFalse(
            $Resource->UserCanViewField(self::$AdminUser, PHP_INT_MAX),
            "Check that users cannot view invalid fields.");

        $Field = $Resource->Schema()->GetField("Test Text Field");
        $Field->Enabled(FALSE);


        # do disabled field check on a copy of the resource so that
        # the PermissionCache doesn't cause it to succeed erroneously
        $RCopy = new Resource($Resource->Id());
        $this->assertFalse(
            $RCopy->UserCanViewField(self::$AdminUser, $Field),
            "Check that users cannot view disabled fields.");
        $Field->Enabled(TRUE);
    }

    /**
    * Check that GetSchemaForResource() returns correct values.
    * @param Resource $Resource Resource to test.
    * @param Resource $RefResource Second resource to test.
    */
    private function CheckGetSchemaForResource($Resource, $RefResource)
    {
        $this->assertEquals(
            MetadataSchema::SCHEMAID_DEFAULT,
            Resource::GetSchemaForResource($Resource->Id()),
             "Check that GetSchemaIdForResource() is correct with a single resource.");

        try
        {
            Resource::GetSchemaForResource(PHP_INT_MAX);
            $this->assertFalse(
                TRUE, "GetSchemaForResource() did not throw exception on invalid Id.");
        }
        catch (Exception $e)
        {
            $this->assertTrue(
                $e instanceof InvalidArgumentException,
                "GetSchemaForResource() threw wrong exception type.");
        }

        $Ids = [$Resource->Id(), $RefResource->Id()];
        $this->assertEquals(
            array_fill_keys($Ids, MetadataSchema::SCHEMAID_DEFAULT),
            Resource::GetSchemaForResource($Ids),
            "Check that GetSchemaIdForResource() is correct with multiple resources.");

        $Ids[]= PHP_INT_MAX;
        try
        {
            Resource::GetSchemaForResource($Ids);
            $this->assertFalse(
                TRUE, "GetSchemaForResource() did not throw exception on invalid Id.");
        }
        catch (Exception $e)
        {
            $this->assertTrue(
                $e instanceof InvalidArgumentException,
                "GetSchemaForResource() threw wrong exception type.");
        }
    }

    /**
    * Check that GetAsArray() returns correct values.
    * @param Resource $Resource Resource to test.
    * @param Resource $RefResource Resource to use as a reference field value.
    */
    private function CheckGetAsArray($Resource, $RefResource)
    {
        $Values = [
            "Test Text Field" => "TestValue",
            "Test Url Field" => "http://example.com",
            "Test Reference Field" =>
                [$RefResource->Id() => $RefResource->Id()],
            "Test User Field" =>
                [$GLOBALS["G_User"]->Id() => $GLOBALS["G_User"]->Get("UserName")],
            "Test Option Field" =>
                [self::$TestOptionCName->Id() => self::$TestOptionCName->Name()],
            "Test CName Field" =>
                [self::$TestControlledName->Id() => self::$TestControlledName->Name()],
            "Test Tree Field" =>
                [self::$TestClassification->Id() => self::$TestClassification->FullName()],
        ];

        foreach ($Values as $FieldName => $Value)
        {
            $Resource->Set($FieldName, $Value);
        }

        $Result = $Resource->GetAsArray(FALSE, FALSE);

        # subset to just the fields that we've set
        $Result = array_intersect_key($Result, $Values);

        $this->assertEquals(
            $Values, $Result, "Checking GetAsArray()");
    }

    /**
    * Check that permanent resources can be made temporary, and that
    * all their field values remain unchanged when that happens.
    * @param Resource $Resource Permanent resource for testing.
    */
    private function CheckTempToggle($Resource)
    {
        $this->assertFalse(
            $Resource->IsTempResource(),
            "Check that provided resource is permanent.");

        $Before = $Resource->GetAsArray(TRUE, FALSE);

        $this->assertTrue(
            $Resource->IsTempResource(TRUE),
            "Check that permanent resources can be made temporary.");

        $After = $Resource->GetAsArray(TRUE, FALSE);
        unset($Before["ResourceId"]);
        unset($After["ResourceId"]);

        $this->assertEquals(
            $Before, $After,
            "Check that resource values don't change on perm/temp toggle");

        $RCopy = new Resource($Resource->Id());

        $AfterCopy = $RCopy->GetAsArray(TRUE, FALSE);
        unset($AfterCopy["ResourceId"]);

        $this->assertEquals(
            $After, $AfterCopy,
            "Check that resource values don't change on perm/temp toggle "
            ."w/ newly loaded resource");
    }

    protected static $TestFieldIds;
    protected static $TestFields;
    protected static $AdminUser;
    protected static $TestClassification;
    protected static $TestControlledName;
    protected static $TestOptionCName;
}
