<?PHP
#
#   FILE:  HtmlRadioButtonSet.php
#
#   Part of the ScoutLib application support library
#   Copyright 2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu
#

/**
* Convenience class for generating a set of HTML radio button form elements.
*/
class HtmlRadioButtonSet
{

    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
    * Class constructor.
    * @param string $ResultVar Name of form variable for select element.
    * @param array $Options Array of options, with form values for the
    *       array index and labels for the array values.
    * @param mixed $SelectedValue Currently selected form value or array
    *       of currently selected form values.  (OPTIONAL)
    */
    public function __construct(
            $ResultVar,
            array $Options,
            $SelectedValue = NULL)
    {
        $this->ResultVar = $ResultVar;
        $this->Options = $Options;
        $this->SelectedValue = $SelectedValue;
    }

    /**
    * Print HTML for set.
    */
    public function PrintHtml()
    {
        print $this->GetHtml();
    }

    /**
    * Get HTML for set.
    * @return string Generated HTML.
    */
    public function GetHtml()
    {
        # start out with empty HTML
        $Html = "";

        # if there are options or we are supposed to print even if no options
        if (count($this->Options) || $this->PrintIfEmpty)
        {
            # for each option
            foreach ($this->Options as $Value => $Label)
            {
                # generate ID for elements
                $Id = $this->ResultVar."_"
                        .preg_replace("%[^a-z0-9]%i", "", $Value);

                # start input element
                $Html .= "    <input type=\"radio\""
                        ." id=\"".htmlspecialchars($Id)."\""
                        ." name=\"".htmlspecialchars($this->ResultVar)."\""
                        ." value=\"".htmlspecialchars($Value)."\"";

                # add in selected attribute if appropriate
                if ((is_array($this->SelectedValue)
                                && in_array($Value, $this->SelectedValue))
                        || ($Value == $this->SelectedValue))
                {
                    $Html .= " checked";
                }

                # add in disabled attribute if appropriate
                if ($this->Disabled
                        || array_key_exists($Value, $this->DisabledOptions))
                {
                    $Html .= " disabled";
                }

                # end input element
                $Html .= ">";

                # add label
                $Html .= " <label for=\"".$Id."\">"
                        .htmlspecialchars($Label)."</label>\n";
            }
        }

        # return generated HTML to caller
        return $Html;
    }

    /**
    * Get/set disabled options.
    * @param mixed $Options Option or array of options to disable.  If
    *       a single option then it should be the value and will be added
    *       to any existing disabled options, and if an array it should have
    *       the values for the index and will replace the current list of
    *       disabled options.  (OPTIONAL)
    */
    public function DisabledOptions($Options = NULL)
    {
        if ($Options !== NULL)
        {
            if (is_array($Options))
            {
                $this->DisabledOptions = $Options;
            }
            else
            {
                $this->DisabledOptions[$Options] = "X";
            }
        }
        return $this->DisabledOptions;
    }

    /**
    * Get/set currently selected value or array of currently selected values.
    * @param mixed $NewValue Currently selected form value or array
    *       of currently selected form values.  (OPTIONAL)
    * @return mixed Selected value or array of currently selected values.
    */
    public function SelectedValue($NewValue = NULL)
    {
        if ($NewValue !== NULL)
        {
            $this->SelectedValue = $NewValue;
        }
        return $this->SelectedValue;
    }

    /**
    * Get/set whether list should be output even if there are no items.
    * If this is set to FALSE and there are no items in the list, GetHtml()
    * will return an empty string and PrintHtml() will print nothing.
    * Defaults to TRUE.
    * @param bool $NewValue If TRUE, HTML will be returned/printed even if
    *       there are no items in the list.  (OPTIONAL)
    * @return bool TRUE if empty list will be printed, otherwise FALSE.
    */
    public function PrintIfEmpty($NewValue = NULL)
    {
        if ($NewValue !== NULL)
        {
            $this->PrintIfEmpty = $NewValue ? TRUE : FALSE;
        }
        return $this->PrintIfEmpty;
    }

    /**
    * Get/set whether the whole option list is editable.  NOTE: When the
    * list is not editable, values for it are not submitted with the form.
    * This is distinct from whether individual options are disabled.
    * @param bool $NewValue If TRUE, list is not editable.
    * @return bool TRUE if list will not be editabled, otherwise FALSE.
    */
    public function Disabled($NewValue = NULL)
    {
        if ($NewValue !== NULL)
        {
            $this->Disabled = $NewValue ? TRUE : FALSE;
        }
        return $this->Disabled;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $Options;
    private $ResultVar;

    private $Disabled = FALSE;
    private $DisabledOptions = array();
    private $PrintIfEmpty = TRUE;
    private $SelectedValue;
}
