<?PHP
#
#   FILE:  Rules.php  (Rules plugin)
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2012-2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis
#

/**
* Plugin for defining and acting upon rules that describe a change in one or
* more metadata field states and actions to take when those changes are detected.
**/
class Rules extends Plugin
{

    # ---- STANDARD PLUGIN INTERFACE -----------------------------------------

    /**
    * Set the plugin attributes.  At minimum this method MUST set $this->Name
    * and $this->Version.  This is called when the plugin is initially loaded.
    */
    public function Register()
    {
        $this->Name = "Rules";
        $this->Version = "2.0.1";
        $this->Description = "Allows specifying rules that describe changes"
                ." in metadata that will trigger email to be sent or other"
                ." actions.";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = array(
                "CWISCore" => "3.9.1");
        $this->EnabledByDefault = TRUE;

        $this->CfgSetup["MinutesBetweenChecks"] = array(
                "Type" => "Number",
                "Label" => "Rule Check Interval",
                "Units" => "minutes",
                "MaxVal" => 999999,
                "Default" => 5,
                "Help" => "The number of minutes between checks for rules"
                        ." that may be ready to execute.  (This does not"
                        ." include rules set to be checked <i>On Change</i>"
                        ." which will be checked any time a resource is"
                        ." updated.)",
                );
    }

    /**
    * Perform any work needed when the plugin is first installed (for example,
    * creating database tables).
    * @return NULL if installation succeeded, otherwise a string containing
    *       an error message indicating why installation failed.
    */
    public function Install()
    {
        # create database tables
        return $this->CreateTables($this->SqlTables);
    }

    /**
    * Perform any work needed when the plugin is uninstalled.
    * @return NULL if uninstall succeeded, otherwise a string containing
    *       an error message indicating why uninstall failed.
    */
    public function Uninstall()
    {
        # drop database tables
        return $this->DropTables($this->SqlTables);
    }

    /**
    * Upgrade from a previous version.
    * @param string $PreviousVersion Previous version of the plugin.
    * @return Returns NULL on success and an error message otherwise.
    */
    public function Upgrade($PreviousVersion)
    {
        if (version_compare($PreviousVersion, "2.0.0", "<"))
        {
            return $this->CreateTables($this->SqlTables);
        }

        if (version_compare($PreviousVersion, "2.0.1", "<"))
        {
            $RFactory = new Rules_RuleFactory();
            foreach ($RFactory->GetItems() as $RuleId => $Rule)
            {
                if ($Rule->Action() == Rules_Rule::ACTION_SENDEMAIL)
                {
                    $ActionParams = $Rule->ActionParameters();
                    if (!array_key_exists("ConfirmBeforeSending", $ActionParams))
                    {
                        $ActionParams["ConfirmBeforeSending"] = FALSE;
                        $Rule->ActionParameters($ActionParams);
                    }
                }
            }
        }

        return NULL;
    }

    /**
    * Hook the events into the application framework.
    * @return Returns an array of events to be hooked into the application
    *      framework.
    */
    public function HookEvents()
    {
        return array(
                "EVENT_COLLECTION_ADMINISTRATION_MENU" => "AddCollectionAdminMenuItems",
                "EVENT_PERIODIC" => "PeriodicRuleCheck",
                "EVENT_RESOURCE_MODIFY" => "ResourceModifiedRuleCheck",
                "EVENT_USER_PRIVILEGES_CHANGED" => "UserPrivChangeRuleUpdate",
                );
    }


    # ---- CALLABLE METHODS --------------------------------------------------

    /**
    * Check rules and take any appropriate corresponding actions.
    */
    public function CheckRules()
    {
        # for each rule ready to be checked
        $RFactory = new Rules_RuleFactory();
        foreach ($RFactory->GetRulesReadyToCheck() as $RuleId => $Rule)
        {
            # check rule for matching items
            $ItemIds = $Rule->Check();

            # if there were items that matched
            if (count($ItemIds))
            {
                # perform rule action on items
                # (Act() includes a LastMatchingIds() update, so we
                #  don't need one here)
                $Rule->Act($ItemIds);
            }
        }
    }

    /**
    * Update matching Ids for a specified user without taking
    * corresponding actions -- typically after user privs are changed.
    * @param int $UserId User to update for.
    */
    public function UpdateMatchingIdsForUser($UserId)
    {
        # if there are search index rebuild tasks running or queued
        if ($GLOBALS["AF"]->TaskIsInQueue(["SPTSearchEngine", "RunUpdateForItem"]))
        {
            # requeue ourselves
            $GLOBALS["AF"]->RequeueCurrentTask();
            return;
        }

        $RFactory = new Rules_RuleFactory();
        foreach ($RFactory->GetItems() as $RuleId => $Rule)
        {
            $NewMatchingIds = $Rule->Check();
            $MatchingIds = $Rule->LastMatchingIds();
            if (isset($NewMatchingIds[$UserId]))
            {
                $MatchingIds[$UserId] = $NewMatchingIds[$UserId];
                $Rule->LastMatchingIds($MatchingIds);
            }
        }
    }

    /**
    * Check rules when run as a background task.
    */
    public function CheckRulesTask()
    {
        # if there are search index rebuild tasks running or queued or
        # if there's an UpdateMatchingIdsForUser task queued (because
        # of user priv update)
        if ($GLOBALS["AF"]->TaskIsInQueue(["SPTSearchEngine", "RunUpdateForItem"]) ||
            $GLOBALS["AF"]->TaskIsInQueue(["Rules", "UpdateMatchingIdsForUser"]) )
        {
            # requeue ourselves
            $GLOBALS["AF"]->RequeueCurrentTask();
        }
        else
        {
            # check rules
            $this->CheckRules();
        }
    }


    # ---- HOOKED METHODS ----------------------------------------------------

    /**
    * Add entries to the Collection Administration menu.
    * @return array List of entries to add, with the label as the value and
    *       the page to link to as the index.
    */
    public function AddCollectionAdminMenuItems()
    {
        return array(
                "ListRules" => "Automation Rules",
                );
    }

    /**
    * Check rules and take any corresponding actions (hooked to EVENT_PERIODIC).
    * @param string $LastRunAt Date and time the event was last run, in SQL
    *       date format. (Passed in by the event signaler, but not used)
    * @return int Number of minutes before the even should be run again.
    */
    public function PeriodicRuleCheck($LastRunAt)
    {
        # check the rules and take any necessary actions
        $this->CheckRulesTask();

        # return to caller the number of minutes before we should check again
        return $this->ConfigSetting("MinutesBetweenChecks");
    }

    /**
    * Queue task to check rules and take any corresponding actions (hooked
    * to EVENT_RESOURCE_MODIFY).
    * @param Resource $Resource The Resource that has been modified.
    *       (passed in from the Event signaler, but not used)
    */
    public function ResourceModifiedRuleCheck($Resource)
    {
        # queue rule check task with (if possible) lower priority than
        #       search index rebuild
        $RuleCheckCallback = array($this, "CheckRulesTask");
        $Priority = min(ApplicationFramework::PRIORITY_BACKGROUND,
                ($GLOBALS["G_SysConfig"]->SearchEngineUpdatePriority() + 1));
        $Description = "Check automation rules after modification of"
                ." <a href=\"r".$Resource->Id()."\"><i>"
                .$Resource->GetMapped("Title")."</i></a>";
        $GLOBALS["AF"]->QueueUniqueTask(
                $RuleCheckCallback, array(), $Priority, $Description);
    }


    /**
    * Queue a task to check all rules but do NOT take corresponding
    * actions (hooked to EVENT_USER_PRIVILEGES_CHANGED).
    * @param int $UserId User whose privileges were updated.
    * @param array $OldPrivileges Privileges user previously had.
    * @param array $NewPrivileges Privileges user has now.
    */
    public function UserPrivChangeRuleUpdate(
        $UserId, $OldPrivileges, $NewPrivileges)
    {
        $RuleCheckCallback = array($this, "UpdateMatchingIdsForUser");
        $Priority = min(ApplicationFramework::PRIORITY_BACKGROUND,
                ($GLOBALS["G_SysConfig"]->SearchEngineUpdatePriority() + 1));
        $Description = "Update last match list for all rules after modification of "
                ."user privileges.";
        $GLOBALS["AF"]->QueueUniqueTask(
                $RuleCheckCallback, array($UserId), $Priority, $Description);


    }

    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $SqlTables = array(
            "Rules" => "CREATE TABLE IF NOT EXISTS Rules_Rules (
                RuleId              INT NOT NULL AUTO_INCREMENT,
                Name                TEXT,
                Enabled             INT DEFAULT 1,
                CheckFrequency      INT DEFAULT 60,
                LastChecked         DATETIME,
                SearchParams        BLOB,
                LastMatchingIds     BLOB,
                Action              INT,
                ActionParams        BLOB,
                DateCreated         DATETIME,
                CreatedBy           INT,
                DateLastModified    DATETIME,
                LastModifiedBy      INT,
                INDEX           Index_I (RuleId)
            );",
            );
}
