<?PHP
#
#   FILE:  RSSImport_Feed.php (RSSImport plugin)
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Class representing an imported feed in the RSSImport plugin.
*/
class RSSImport_Feed extends Item
{
    # ---- PUBLIC INTERFACE --------------------------------------------------

    /** Display Locations. */
    const DISPLOC_ALLPAGES  = 1;
    const DISPLOC_FRONTPAGE = 2;
    const DISPLOC_ADMINPAGE = 4;
    const DISPLOC_MTOOLPAGE = 8;

    /** Refresh Rates. */
    const REFRATE_86400 = "86400";
    const REFRATE_43200 = "43200";
    const REFRATE_21600 = "21600";
    const REFRATE_14400 = "14400";
    const REFRATE_3600  = "3600";
    const REFRATE_1800  = "1800";
    const REFRATE_900   = "900";
    const REFRATE_600   = "600";
    const REFRATE_300   = "300";
    const REFRATE_120   = "120";
    const REFRATE_60    = "60";

    /**
    * Create new feed item.
    * @return object New feed object.
    */
    public static function Create()
    {
        # instantiate new Feed object
        $InitialValues = array(
                "Enabled" => TRUE,
                );
        $Feed = parent::CreateWithValues($InitialValues);

        # return new Feed object to caller
        return $Feed;
    }

    /**
    * Get/set whether the feed is enabled.
    * @param bool $NewValue TRUE to enable, or FALSE to disable. (OPTIONAL)
    * @return bool TRUE if feed is enabled, otherwise FALSE.
    */
    public function Enabled($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("Enabled", $NewValue);
    }

    /**
    * Get/set the feed title.
    * @param string $NewValue New title of the feed. (OPTIONAL)
    * @return string value of the feed title.
    */
    public function Title($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("Title", $NewValue);
    }

    /**
    * Get/set the feed URL.
    * @param string $NewValue New URL of the feed. (OPTIONAL)
    * @return string value of the feed URL.
    */
    public function Url($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("URL", $NewValue);
    }

    /**
    * Get/set feed display locations.
    * @param int $NewValue Value corresponding to new display location. (OPTIONAL)
    * @return int value of the display location.
    */
    public function DisplayLocations($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("DisplayLocations", $NewValue);
    }

    /**
    * Get/set number of feed items to display.
    * @param int $NewValue New number of feeds to display. (OPTIONAL)
    * @return int number of feeds to display.
    */
    public function NumberItemsToDisplay($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("NumberItemsToDisplay", $NewValue);
    }

    /**
    * Get/set how often the feed updates.
    * @param int $NewValue New refresh rate. (OPTIONAL)
    * @return int refresh rate of the feed.
    */
    public function RefreshRate($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("RefreshRate", $NewValue);
    }

    /**
    * @return an array of possible RSS feed display locations
    */
    public static function PossibleDisplayLocations()
    {
        return array(
            self::DISPLOC_ALLPAGES => "All Pages",
            self::DISPLOC_FRONTPAGE => "Front Page",
            self::DISPLOC_ADMINPAGE => "Portal Administration Page",
            self::DISPLOC_MTOOLPAGE => "Metadata Tool Page",
            );
    }

    /**
    * @return an array of possible RSS feed refresh rates
    */
    public static function PossibleRefreshRates()
    {
        return array(
            self::REFRATE_86400 => "Daily",
            self::REFRATE_43200 => "12 Hours",
            self::REFRATE_21600 => "6 Hours",
            self::REFRATE_14400 => "4 Hours",
            self::REFRATE_3600  => "1 Hour",
            self::REFRATE_1800  => "30 Minutes",
            self::REFRATE_900   => "15 Minutes",
            self::REFRATE_600   => "10 Minutes",
            self::REFRATE_300   => "5 Minutes",
            self::REFRATE_120   => "2 Minutes",
            self::REFRATE_60    => "1 Minute",
            );
    }

    # ---- PRIVATE INTERFACE -------------------------------------------------

    /**
    * Set the database access values (table name, ID column name, name column
    * name) for specified class.  This may be overridden in a child class, if
    * different values are needed.
    * @param string $ClassName Class to set values for.
    */
    static protected function SetDatabaseAccessValues($ClassName)
    {
        if (!isset(self::$ItemIdColumnNames[$ClassName]))
        {
            self::$ItemIdColumnNames[$ClassName] = "FeedId";
            self::$ItemNameColumnNames[$ClassName] = "Title";
            self::$ItemTableNames[$ClassName] = "RSSImport_Feeds";
        }
    }
}
