<?PHP
#
#   FILE:  DrupalSync.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2002-2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Plugin to synchronize logins between a CWIS and Drupal installation.
*
* A companion cwis_user plugin must also be installed on the target drupal site.
*/
final class DrupalSync extends Plugin
{
    /**
    * Register the DrupalSync plugin.
    * @see Plugin::Register
    */
    public function Register()
    {
        $this->Name = "Drupal Login Synchronization";
        $this->Version = "2.0.0";
        $this->Description = "Allows users to register or log in"
            ." on CWIS or an associated Drupal installation "
            ." and be registered or logged in on the other package as well.  "
            ." An included companion module (<i>cwis_user</i>) must"
            ." be loaded on the Drupal site for this plugin to work";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = array(
            "CWISCore" => "3.9.2"
            );

        $this->CfgSetup["DrupalEndpoint"] = array(
            "Type" => "URL",
            "Label" => "Drupal URL",
            "Help" => " The URL of your Drupal site, not including 'index.php "
                ." (e.g., https://example.com/Drupal/) .",
        );

        $this->CfgSetup["Password"] = array(
            "Type" => "Text",
            "Label" => "Synchronization Password",
            "Help" => "Password used to encrypt login synchronization messages. "
                ."Must be the same here as in the cwis_user module in the Drupal site."
        );
    }

    /**
    * Initialize plugin.
    * @see Plugin::Initialize
    */
    public function Initialize()
    {
        # DrupalSite/cwis_user/api is created in the Drupal site by
        # the cwis_user plugin
        $this->RESTHelper = new RestAPIHelper(
            $this->ConfigSetting("DrupalEndpoint")."/cwis_user/api",
            $this->ConfigSetting("Password"),
            array($this, "IsMessageDuplicate"),
            array($this, "RegisterMessage") );

        $this->DB = new Database();
    }

    /**
    * Install plugin.
    * @see Plugin::Install
    */
    public function Install()
    {
        return $this->CreateTables($this->SqlTables);
    }

    /**
    * Uninstall plugin.
    * @see Plugin::Uninstall.
    */
    public function Uninstall()
    {
        return $this->DropTables($this->SqlTables);
    }

    /**
    * Upgrade the DrupalSync plugin.
    * @param string $PreviousVersion Previously installed version of the plugin.
    * @see Plugin::Upgrade
    */
    public function Upgrade($PreviousVersion)
    {
        $DB = new Database();
        if (version_compare($PreviousVersion, "1.0.2", "<"))
        {
            foreach (array("DrupalToCwisUserSync" => "DrupalSync_DtoC",
                           "CwisToDrupalUserSync" => "DrupalSync_CtoD",
                           "UserNameMap"          => "DrupalSync_UserNameMap"
            ) as $OldName => $NewName )
            {
                $DB->Query("ALTER TABLE ".$OldName
                           ." RENAME TO ".$NewName);
            }
        }
        if (version_compare($PreviousVersion, "2.0.0", "<"))
        {
            foreach (array("DtoC", "CtoD", "UserNameMap") as $Table)
            {
                $DB->Query("DROP TABLE DrupalSync_".$Table);
            }

            $DB->Query(
                "CREATE TABLE DrupalSync_SeenMessages (Timestamp INT, Cookie TEXT)");
        }
    }

    /**
    * Hook the DrupalSync plugin into the event system.
    * @see Plugin::HookEvents
    */
    public function HookEvents()
    {
        return array(
            "EVENT_USER_ADDED"            => "DrupalMirrorUserAdd",
            "EVENT_USER_DELETED"          => "DrupalMirrorUserDel",
            "EVENT_USER_PASSWORD_CHANGED" => "DrupalMirrorUserPasswordUpdate",
            "EVENT_USER_EMAIL_CHANGED"    => "DrupalMirrorUserEmailUpdate",
            "EVENT_USER_LOGIN"            => "DrupalMirrorUserLogin",
            "EVENT_USER_LOGOUT"           => "DrupalMirrorUserLogout"
            );
    }

    /**
    * Tell the Druapl site about user creations.
    * @param int $UserId Id of newly added user.
    * @param string $Password Password of newly added user.
    */
    public function DrupalMirrorUserAdd($UserId, $Password)
    {
        $User = new SPTUser($UserId);
        if (!is_object($User) || $User->Status()!=U_OKAY)
        {
            return;
        }

        $this->RESTHelper->DoRestCommand(array(
            "Cmd" => "CreateUser",
            "UserName" => $User->Name(),
            "UserEmail" => $User->Get("EMail"),
            "Password" => $Password,
        ));

    }

    /**
    * Tell the Druapl site about user deletions.
    * @param int $UserId Id of user to be deleted.
    */
    public function DrupalMirrorUserDel($UserId)
    {
        $User = new SPTUser($UserId);
        if (!is_object($User) || $User->Status()!=U_OKAY)
        {
            return;
        }

        # bail if we have multiple users with this email address
        if ($this->CountUsersWithEmail($User->Get("EMail")) > 1)
        {
            return;
        }

        $this->RESTHelper->DoRestCommand(array(
            "Cmd" => "DeleteUser",
            "UserEmail" => $User->Get("EMail"),
        ));
    }

    /**
    * Tell the Druapl site about updates to user passwords.
    * @param int $UserId Id of user whose password was changed.
    * @param string $OldPass Old password.
    * @param string $NewPass New password.
    */
    public function DrupalMirrorUserPasswordUpdate($UserId, $OldPass, $NewPass)
    {
        $User = new SPTUser($UserId);
        if (!is_object($User) || $User->Status()!=U_OKAY)
        {
            return;
        }

        $this->RESTHelper->DoRestCommand(array(
            "Cmd" => "UpdatePassword",
            "UserEmail" => $User->Get("EMail"),
            "UserPassword" => $OldPass,
            "NewPassword" => $NewPass));
    }

    /**
    * Tell the Drupal site about updates to user email.
    * @param int $UserId Id of user whose email was changed.
    * @param string $OldEmail Old email address.
    * @param string $NewEmail New email address.
    */
    public function DrupalMirrorUserEmailUpdate($UserId, $OldEmail, $NewEmail)
    {
        $User = new SPTUser($UserId);
        if (!is_object($User) || $User->Status()!=U_OKAY)
        {
            return;
        }

        # bail if we have multiple users with this email address
        if ($this->CountUsersWithEmail($User->Get("EMail")) > 1)
        {
            return;
        }

        $this->RESTHelper->DoRestCommand(array(
            "Cmd" => "UpdateEmail",
            "UserEmail" => $OldEmail,
            "NewEmail" => $NewEmail));
    }

    /**
    * Tell the Drupal site when users log in.
    * @param int $UserId Id of user who logged in.
    * @param string $Password User's password.
    */
    public function DrupalMirrorUserLogin($UserId, $Password)
    {
        $User = new SPTUser($UserId);
        if (!is_object($User) || $User->Status()!=U_OKAY)
        {
            return;
        }

        $SName = session_name();
        $Cookies = array(
            $SName => session_id(),
        );

        $Result = $this->RESTHelper->DoRestCommand(array(
            "Cmd" => "UserLogin",
            "UserEmail" => $User->Get("EMail"),
            "UserPassword" => $Password,
            "Cookies" => $Cookies,
        ));

        # bail on failure
        if ($Result["Status"] != "OK" ||
            $Result["Message"] != "Login successful.")
        {
            return;
        }

        $Hostname = parse_url(
            $this->ConfigSetting("DrupalEndpoint"), PHP_URL_HOST);
        $Proto = parse_url(
            $this->ConfigSetting("DrupalEndpoint"), PHP_URL_SCHEME);

        foreach ($Result["Cookies"] as $Name => $Val)
        {
            setcookie($Name, $Val, time() + 3600, "/", $Hostname,
                      ($Proto == "https" ? TRUE : FALSE), TRUE);
        }

        $_SESSION["DrupalSync_NeedsLogout"] = $Result["Cookies"];
    }

    /**
    * Tell the Druapl site when users log out.
    * @param int $UserId Id of user who logged out.
    */
    public function DrupalMirrorUserLogout($UserId)
    {
        $User = new SPTUser($UserId);
        if (!is_object($User) || $User->Status()!=U_OKAY)
        {
            return;
        }

        if (!isset($_SESSION["DrupalSync_NeedsLogout"]))
        {
            return;
        }

        $Result = $this->RESTHelper->DoRestCommand(array(
            "Cmd" => "UserLogout",
            "UserEmail" => $User->Get("EMail"),
            "Cookies" => $_SESSION["DrupalSync_NeedsLogout"],
        ));

        if ($Result["Status"] != "OK" ||
            $Result["Message"] != "User successfully logged out.")
        {
            return;
        }

        $Hostname = parse_url(
            $this->ConfigSetting("DrupalEndpoint"), PHP_URL_HOST);
        $Proto = parse_url(
            $this->ConfigSetting("DrupalEndpoint"), PHP_URL_SCHEME);

        foreach ($_SESSION["DrupalSync_NeedsLogout"] as $Name => $Val)
        {
            setcookie($Name, NULL, time() - 3600, "/", $Hostname,
                      ($Proto == "https" ? TRUE : FALSE), TRUE);
        }

        unset($_SESSION["DrupalSync_NeedsLogout"]);
    }

    /**
    * Register a REST API message as having been seen.
    * @param int $Timestamp Timestamp of message.
    * @param string $Cookie Random nonce in message.
    */
    public function RegisterMessage($Timestamp, $Cookie)
    {
        $this->DB->Query(
            "INSERT INTO DrupalSync_SeenMessages (Timestamp, Cookie) "
            ."VALUES (".intval($Timestamp).",'".addslashes($Cookie)."')");
    }

    /**
    * Determine if a specific REST API message has already been seen before.
    * @param int $Timestamp Timestamp of message.
    * @param string $Cookie Random nonce in message.
    * @return bool TRUE for duplicate messages.
    */
    public static function IsMessageDuplicate($Timestamp, $Cookie)
    {
        $N = $this->DB->Query(
            "SELECT COUNT(*) AS N FROM DrupalSync_SeenMessages "
            ."WHERE Timestamp=".intval($Timestamp)." AND "
            ."Cookie='".addslashes($Cookie)."'", "N");

        return ($N > 0);
    }

    /**
    * Access RestAPI Helper.
    * @return RestAPI Helper.
    */
    public function RESTHelper()
    {
        return $this->RESTHelper;
    }

    /**
    * Count the number of local users with a specified email address.
    * @param string $EMail The email to search for.
    * @return int count of users having that email.
    */
    private function CountUsersWithEmail($EMail)
    {
        $UFactory = new CWUserFactory();
        $Matches = $UFactory->GetMatchingUsers(
            $EMail, "EMail");
        return count($Matches);
    }

    private $SqlTables = array(
        "SeenMessages" => "CREATE TABLE IF NOT EXISTS
            DrupalSync_SeenMessages (Timestamp INT, Cookie TEXT);");
    private $DB;
    private $RESTHelper;
}
