<?PHP

#
#   FILE:  Email.php
#
#   Part of the ScoutLib application support library
#   Copyright 2012 Edward Almasy and Internet Scout
#   http://scout.wisc.edu
#

/**
* Electronic mail message.
* \nosubgrouping
*/
class Email {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
    * Object constructor.
    */
    function __construct()
    {
    }


    /** @name Sending */ /*@(*/

    /**
    * Mail the message.
    * @return TRUE if message was successfully accepted for delivery,
    *       otherwise FALSE.
    */
    function Send()
    {
        switch (self::$DeliveryMethod)
        {
            case self::METHOD_PHPMAIL:
                $Result = $this->SendViaPhpMail();
                break;

            case self::METHOD_SMTP:
                $Result = $this->SendViaSmtp();
                break;
        }
        return $Result;
    }


    /** @name Message Attributes */ /*@(*/

    /**
    * Get/set message body.
    * @param NewValue New message body.  (OPTIONAL)
    * @return Current message body.
    */
    function Body($NewValue = NULL)
    {
        if ($NewValue !== NULL) {  $this->Body = $NewValue;  }
        return $this->Body;
    }

    /**
    * Get/set message subject.
    * @param NewValue New message subject.  (OPTIONAL)
    * @return Current message subject.
    */
    function Subject($NewValue = NULL)
    {
        if ($NewValue !== NULL) {  $this->Subject = $NewValue;  }
        return $this->Subject;
    }

    /**
    * Get/set message sender.
    * @param NewAddress New message sender address.  (OPTIONAL, but required if
    *       NewName is specified.)
    * @param NewName New message sender name.  (OPTIONAL)
    * @return Current message sender in RFC-2822 format ("user@example.com" or
    *       "User <user@example.com>" if name available).
    */
    function From($NewAddress = NULL, $NewName = NULL)
    {
        if ($NewAddress !== NULL)
        {
            $NewAddress = trim($NewAddress);
            if ($NewName !== NULL)
            {
                $NewName = trim($NewName);
                $this->From = $NewName." <".$NewAddress.">";
            }
            else
            {
                $this->From = $NewAddress;
            }
        }
        return $this->From;
    }

    /**
    * Get/set message "Reply-To" address.
    * @param NewAddress New message "Reply-To" address.  (OPTIONAL, but required if
    *       NewName is specified.)
    * @param NewName New message "Reply-To" name.  (OPTIONAL)
    * @return Current message "Reply-To" address in RFC-2822 format
    *       ("user@example.com" or "User <user@example.com>" if name available).
    */
    function ReplyTo($NewAddress = NULL, $NewName = NULL)
    {
        if ($NewAddress !== NULL)
        {
            $NewAddress = trim($NewAddress);
            if ($NewName !== NULL)
            {
                $NewName = trim($NewName);
                $this->ReplyTo = $NewName." <".$NewAddress.">";
            }
            else
            {
                $this->ReplyTo = $NewAddress;
            }
        }
        return $this->ReplyTo;
    }

    /**
    * Get/set message recipient(s).
    * @param NewValue New message recipient or array of recipients, in
    *       RFC-2822 format ("user@example.com" or "User <user@example.com>"
    *       if name included).  (OPTIONAL)
    * @return Array of current message recipient(s) in RFC-2822 format.
    */
    function To($NewValue = NULL)
    {
        if ($NewValue !== NULL)
        {
            if (!is_array($NewValue))
            {
                $this->To = array($NewValue);
            }
            else
            {
                $this->To = $NewValue;
            }
        }
        return $this->To;
    }

    /**
    * Get/set message CC list.
    * @param NewValue New message CC recipient or array of CC recipients, in
    *       RFC-2822 format ("user@example.com" or "User <user@example.com>"
    *       if name included).  (OPTIONAL)
    * @return Array of current message CC recipient(s) in RFC-2822 format.
    */
    function CC($NewValue = NULL)
    {
        if ($NewValue !== NULL)
        {
            if (!is_array($NewValue))
            {
                $this->CC = array($NewValue);
            }
            else
            {
                $this->CC = $NewValue;
            }
        }
        return $this->CC;
    }

    /**
    * Get/set message BCC list.
    * @param NewValue New message BCC recipient or array of BCC recipients, in
    *       RFC-2822 format ("user@example.com" or "User <user@example.com>"
    *       if name included).  (OPTIONAL)
    * @return Array of current message BCC recipient(s) in RFC-2822 format.
    */
    function BCC($NewValue = NULL)
    {
        if ($NewValue !== NULL)
        {
            if (!is_array($NewValue))
            {
                $this->BCC = array($NewValue);
            }
            else
            {
                $this->BCC = $NewValue;
            }
        }
        return $this->BCC;
    }

    /**
    * Specify additional message headers to be included.
    * @param NewHeaders Array of header lines.
    */
    function AddHeaders($NewHeaders)
    {
        # add new headers to list
        $this->Headers = array_merge($this->Headers, $NewHeaders);
    }


    /** @name Mail Delivery Method */ /*@(*/

    /**
    * Get/set mail delivery method.  If specified, the method must be one of
    * the predefined "METHOD_" constants.
    * @param NewValue New delivery method.  (OPTIONAL)
    * @return Current delivery method.
    */
    static function DeliveryMethod($NewValue = NULL)
    {
        if ($NewValue !== NULL)
        {
            self::$DeliveryMethod = $NewValue;
        }
        return self::$DeliveryMethod;
    }
    const METHOD_PHPMAIL = 1;
    const METHOD_SMTP = 2;

    /**
    * Get/set server for mail delivery.
    * @param NewValue New server.  (OPTIONAL)
    * @return Current server.
    */
    static function Server($NewValue = NULL)
    {
        if ($NewValue !== NULL) {  self::$Server = $NewValue;  }
        return self::$Server;
    }

    /**
    * Get/set port number for mail delivery.
    * @param NewValue New port number.  (OPTIONAL)
    * @return Current port number.
    */
    static function Port($NewValue = NULL)
    {
        if ($NewValue !== NULL) {  self::$Port = $NewValue;  }
        return self::$Port;
    }

    /**
    * Get/set user name for mail delivery.
    * @param NewValue New user name.  (OPTIONAL)
    * @return Current user name.
    */
    static function UserName($NewValue = NULL)
    {
        if ($NewValue !== NULL) {  self::$UserName = $NewValue;  }
        return self::$UserName;
    }

    /**
    * Get/set password for mail delivery.
    * @param NewValue New password.  (OPTIONAL)
    * @return Current password.
    */
    static function Password($NewValue = NULL)
    {
        if ($NewValue !== NULL) {  self::$Password = $NewValue;  }
        return self::$Password;
    }

    /**
    * Get/set whether to use authentication for mail delivery.
    * @param NewValue New authentication setting.  (OPTIONAL)
    * @return Current authentication setting.
    */
    static function UseAuthentication($NewValue = NULL)
    {
        if ($NewValue !== NULL) {  self::$UseAuthentication = $NewValue;  }
        return self::$UseAuthentication;
    }

    /**
    * Get/set serialized (opaque text) version of delivery settings.  This
    * method is intended to be used to store and retrieve all email delivery
    * settings for the class, in a form suitable to be saved to a database.
    * @param NewSettings New delivery settings values.
    * @return Current delivery settings values.
    */
    static function DeliverySettings($NewSettings = NULL)
    {
        if ($NewSettings !== NULL)
        {
            $Settings = unserialize($NewSettings);
            self::$DeliveryMethod = $Settings["DeliveryMethod"];
            self::$Server = $Settings["Server"];
            self::$Port = $Settings["Port"];
            self::$UserName = $Settings["UserName"];
            self::$Password = $Settings["Password"];
            self::$UseAuthentication = $Settings["UseAuthentication"];
        }
        else
        {
            $Settings["DeliveryMethod"] = self::$DeliveryMethod;
            $Settings["Server"] = self::$Server;
            $Settings["Port"] = self::$Port;
            $Settings["UserName"] = self::$UserName;
            $Settings["Password"] = self::$Password;
            $Settings["UseAuthentication"] = self::$UseAuthentication;
        }
        return serialize($Settings);
    }

    /**
    * Test delivery settings and report their validity.  For example, if
    * the deliver method is set to SMTP it would test the server, port,
    * and (if authentication is indicated) user name and password.  If
    * delivery settings are not okay, then DeliverySettingErrors() can be
    * used to determine (if known) which settings may have problems.
    * @return TRUE if delivery settings are okay, otherwise FALSE.
    */
    static function DeliverySettingsOkay()
    {
        # start out with error list clear
        self::$DeliverySettingErrorList = array();

        # test based on delivery method
        switch (self::$DeliveryMethod)
        {
            case self::METHOD_PHPMAIL:
                # always report success
                $SettingsOkay = TRUE;
                break;

            case self::METHOD_SMTP:
                # set up PHPMailer for test
                $PMail = new PHPMailer(TRUE);
                $PMail->IsSMTP();
                $PMail->SMTPAuth = self::$UseAuthentication;
                $PMail->Host = self::$Server;
                $PMail->Port = self::$Port;
                $PMail->Username = self::$UserName;
                $PMail->Password = self::$Password;

                # test settings
                try
                {
                    $SettingsOkay = $PMail->SmtpConnect();
                }
                # if test failed
                catch (phpmailerException $Except)
                {
                    # translate PHPMailer error message to possibly bad settings
                    switch ($Except->getMessage())
                    {
                        case 'SMTP Error: Could not authenticate.':
                            self::$DeliverySettingErrorList = array(
                                    "UseAuthentication",
                                    "UserName",
                                    "Password",
                                    );
                            break;

                        case 'SMTP Error: Could not connect to SMTP host.':
                            self::$DeliverySettingErrorList = array(
                                    "Server",
                                    "Port",
                                    );
                            break;

                        case 'Language string failed to load: tls':
                            self::$DeliverySettingErrorList = array("TLS");
                            break;

                        default:
                            self::$DeliverySettingErrorList = array("UNKNOWN");
                            break;
                    }

                    # make sure failure is reported
                    $SettingsOkay = FALSE;
                }
                break;
        }

        # report result to caller
        return $SettingsOkay;
    }

    /**
    * Return array with list of delivery setting errors (if any).
    * @return Array with settings that are possibly bad.
    */
    static function DeliverySettingErrors()
    {
        return self::$DeliverySettingErrorList;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $From = "";
    private $ReplyTo = "";
    private $To = array();
    private $CC = array();
    private $BCC = array();
    private $Body = "";
    private $Subject = "";
    private $Headers = array();

    private static $DeliveryMethod = self::METHOD_PHPMAIL;
    private static $DeliverySettingErrorList = array();
    private static $Server;
    private static $Port = 25;
    private static $UserName = "";
    private static $Password = "";
    private static $UseAuthentication = FALSE;

    private function SendViaPhpMail()
    {
        # build headers list
        $Headers = "From: ".self::CleanHeaderValue($this->From)."\r\n";
        $Headers .= $this->BuildAddresseeLine("Cc", $this->CC);
        $Headers .= $this->BuildAddresseeLine("Bcc", $this->BCC);
        $Headers .= "Reply-To: ".self::CleanHeaderValue(
                strlen($this->ReplyTo) ? $this->ReplyTo : $this->From)
                ."\r\n";
        foreach ($this->Headers as $ExtraHeader)
        {
            $Headers .= $ExtraHeader."\r\n";
        }

        # build recipient list
        $To = "";
        $Separator = "";
        foreach ($this->To as $Recipient)
        {
            $To .= $Separator.$Recipient;
            $Separator = ", ";
        }

        # send message
        $Result = mail($To, $this->Subject, $this->Body, $Headers);

        # report to caller whether attempt to send succeeded
        return $Result;
    }

    private function SendViaSmtp()
    {
        # create PHPMailer and set up for SMTP
        $PMail = new PHPMailer();
        $PMail->IsSMTP();
        $PMail->SMTPAuth = self::$UseAuthentication;
        $PMail->Host = self::$Server;
        $PMail->Port = self::$Port;
        $PMail->Username = self::$UserName;
        $PMail->Password = self::$Password;

        # set message attributes
        if (preg_match("/ </", $this->From))
        {
            $Pieces = explode(" ", $this->From);
            $Address = array_pop($Pieces);
            $Address = preg_replace("/[<>]+/", "", $Address);
            $Name = trim(implode($Pieces));
        }
        else
        {
            $Name = "";
            $Address = $this->From;
        }
        $PMail->SetFrom($Address, $Name);
        $PMail->Subject = $this->Subject;
        $PMail->Body = $this->Body;
        $PMail->IsHTML(FALSE);
        foreach ($this->To as $Recipient)
        {
            $PMail->AddAddress($Recipient);
        }

        # add any extra header lines to message
        foreach ($this->Headers as $ExtraHeader)
        {
            $PMail->AddCustomHeader($ExtraHeader);
        }

        # send message
        $Result = $PMail->Send();

        # report to caller whether attempt to send succeeded
        return $Result;
    }

    private function BuildAddresseeLine($Label, $Recipients)
    {
        $Line = "";
        if (count($Recipients))
        {
            $Line .= $Label.": ";
            $Separator = "";
            foreach ($Recipients as $Recipient)
            {
                $FullBody .= $Separator.self::CleanHeaderValue($Recipient);
                $Separator = ", ";
            }
            $Line .= "\r\n";
        }
        return $Line;
    }

    /**
    * Remove problematic content from values to be used in message header.
    * @param Value Value to be sanitized.
    * @return Sanitized value.
    */
    private static function CleanHeaderValue($Value)
    {
        # (regular expression taken from sanitizeHeaders() function in
        #       Mail PEAR package)
        return preg_replace('=((<CR>|<LF>|0x0A/%0A|0x0D/%0D|\\n|\\r)\S).*=i',
                "", $Value);
    }

}


