<?PHP
#
#   FILE:  CWPlugin.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2016 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* This class extends the base Plugin class with CWIS-specific functionality.
*/
abstract class CWPlugin extends Plugin
{
    # ---- PUBLIC INTERFACE --------------------------------------------------


    # ---- PROTECTED INTERFACE -----------------------------------------------

    /**
    * Load fields into metadata schema from XML file.  The XML file is
    * assumed to be in install/MetadataSchema--SCHEMANAME.xml under the
    * plugin's directory.
    * @param mixed $Schema Schema or ID of schema to load fields into.
    * @return string Error message or NULL if load succeeded.
    * @throws Exception If no XML file found.
    */
    protected function AddMetadataFieldsFromXml($Schema)
    {
        # load schema
        if (!is_object($Schema))
        {
            $Schema = new MetadataSchema($Schema);
        }

        # assemble XML file name
        $PossibleSuffixes = array(
                StdLib::Singularize($Schema->ResourceName()),
                StdLib::Pluralize($Schema->ResourceName()));
        foreach ($PossibleSuffixes as $Suffix)
        {
            $FileName = "plugins/".get_class($this)
                    ."/install/MetadataSchema--"
                    .str_replace(" ", "", $Suffix).".xml";
            if (is_file($FileName))
            {
                $XmlFile = $FileName;
                break;
            }
        }
        if (!isset($XmlFile))
        {
            throw new Exception("No XML file found to load metadata fields from"
                    ." for ".$SchemaName." schema.");
        }

        # load fields from file
        $Result = $Schema->AddFieldsFromXmlFile($XmlFile);

        # if load failed
        if ($Result === FALSE)
        {
            # return error message(s) to caller
            return "Error loading User metadata fields from XML: ".implode(
                    " ", $Schema->ErrorMessages("AddFieldsFromXmlFile"));
        }

        # report success to caller
        return NULL;
    }

    /**
    * Delete any metadata fields owned by plugin from specified schema.
    * @param int $SchemaId ID of schema to drop fields from.
    * @return string Error message or NULL if drop succeeded.
    */
    protected function DeleteMetadataFields($SchemaId)
    {
        # load schema
        $Schema = new MetadataSchema($SchemaId);

        # for each field in schema
        foreach ($Schema->GetFields() as $FieldId => $Field)
        {
            # drop field if we own it
            if ($Field->Owner() == get_class($this))
            {
                $Schema->DropField($FieldId);
            }
        }

        # report success to caller
        return NULL;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------
}
