<?PHP
#
#   FILE:  Blog_EntryFactory.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2015 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Factory for BlogEntry objects.
*/
class Blog_EntryFactory extends ResourceFactory
{
    /**
    * Create an EntryFactory to manipulate entries for a particular blog.
    * @param int $BlogId BlogId for this factory.
    * @return object new Blog_EntryFactory object
    */
    public function __construct($BlogId)
    {
        # snag the blog plugin
        $BlogPlugin = $GLOBALS["G_PluginManager"]->GetPlugin("Blog");

        # construct a ResourceFactory for the blog schema
        parent::__construct($BlogPlugin->GetSchemaId());
        # record which ControlledName corresponds with our blog
        $this->BlogCName = new ControlledName( $BlogId );
    }

    /**
    * List all the ItemIds that belong to this blog.
    * @param string $Condition SQL Condition to match.
    * @param bool $IncludeTempItems TRUE to include temp items (OPTIONAL).
    * @param string $SortField Field to sort by (OPTIONAL).
    * @param bool $SortAscending TRUE to sort ascending (OPTIONAL).
    */
    public function GetItemIds($Condition = NULL, $IncludeTempItems = FALSE,
                        $SortField = NULL, $SortAscending = TRUE)
    {
        return array_intersect(
            parent::GetItemIds($Condition, $IncludeTempItems,
                               $SortField, $SortAscending),
            $this->BlogCName->GetAssociatedResources() );
    }

    /**
    * List ResourceIds for this blog, sorted by a specified field.
    * @param string $FieldName Field to sort by.
    * @param bool $Ascending TRUE to sort ascending (OPTIONAL).
    * @param int|null $Limit Number of items to return (OPTIONAL).
    */
    public function GetResourceIdsSortedBy($FieldName, $Ascending = TRUE, $Limit = NULL)
    {
        $Matches = array_intersect(
            parent::GetResourceIdsSortedBy($FieldName, $Ascending),
            $this->BlogCName->GetAssociatedResources() );

        return array_slice($Matches, 0, $Limit);
    }

    /**
    * Find blog entries with values that match specified fields.
    * @param array $ValuesToMatch Array of values to search for, keyed by Field.
    * @param bool $AllRequired TRUE if all values must match (OPTIONAL, default TRUE).
    * @param bool $ReturnObjects TRUE to return objects (OPTIONAL, default TRUE).
    * @param string $Operator Match operator (OPTIONAL, default "==").
    */
    public function GetMatchingResources(
        $ValuesToMatch, $AllRequired=TRUE, $ReturnObjects=TRUE, $Operator="==")
    {
        return array_intersect(
            parent::GetMatchingResources(
                $ValuesToMatch, $AllRequired, $ReturnObjects, $Operator),
            $this->BlogCName->GetAssociatedResources() );
    }

    private $BlogCName;
}
