<?PHP
#
#   FILE:  FileFactory.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2007-2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Factory for manipulating File objects.
*/
class FileFactory extends ItemFactory
{

    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
    * Object constructor.
    * @param int $FieldId Metadata field ID.  (OPTIONAL)
    */
    public function __construct($FieldId = NULL)
    {
        # save field ID for our later use
        $this->FieldId = $FieldId;

        # set up item factory base class
        parent::__construct("File", "Files", "FileId", "FileName", FALSE,
                ($FieldId ? "FieldId = ".intval($FieldId) : NULL));
    }

    /**
    * Retrieve all files (names or objects) for specified resource.
    * @param int|object $ResourceOrResourceId Resource object or ID.
    * @param bool $ReturnObjects Whether to return File objects instead of names.
    * @return array Array of name strings or File objects, with file IDs
    *       for index.  (OPTIONAL, defaults to TRUE)
    */
    public function GetFilesForResource($ResourceOrResourceId, $ReturnObjects = TRUE)
    {
        # start out assuming that no files will be found
        $ReturnValue = array();

        # sanitize resource ID or grab it from object
        $ResourceOrResourceId = is_object($ResourceOrResourceId)
                ? $ResourceOrResourceId->Id() : intval($ResourceOrResourceId);

        # retrieve names and IDs of files associated with resource
        $this->DB->Query(
            "SELECT FileId, FileName FROM Files"
            ." WHERE ResourceId = ".$ResourceOrResourceId
            ." AND FieldId"
            .($this->FieldId ? "=".$this->FieldId : ">0"));
        $FileNames = $this->DB->FetchColumn("FileName", "FileId");

        # if files were found
        if (count($FileNames))
        {
            # if caller asked us to return objects
            if ($ReturnObjects)
            {
                # for each file
                foreach ($FileNames as $FileId => $FileName)
                {
                    # create file object and add it to array
                    $ReturnValue[$FileId] = new File($FileId);
                }
            }
            else
            {
                # return array of file names with IDs as index
                $ReturnValue = $FileNames;
            }
        }

        # return resulting array of files or file names to caller
        return $ReturnValue;
    }

    /**
    * Create copy of File and return to caller.
    * @param object $FileToCopy File object for file to copy.
    * @return object New File object.
    */
    public function Copy($FileToCopy)
    {
        return new File($FileToCopy->GetNameOfStoredFile(),
                $FileToCopy->ResourceId(),
                $FileToCopy->FieldId(),
                $FileToCopy->Name());
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $FieldId;
}
