<?PHP
#
#   FILE:  SysConfig.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

# ----- LOCAL FUNCTIONS ------------------------------------------------------

/**
* Define fields for form.
* @return array Associative array of form field parameters, in the format
*       expected by FormUI.
*/
function DefineFormFields()
{
    # load up possible values for SavedSearchMailTemplate setting
    $SavedSearchMailTemplates = array(-1 => "Default");
    if ($GLOBALS["G_PluginManager"]->PluginEnabled("Mailer"))
    {
        $SavedSearchMailTemplates += $GLOBALS["G_PluginManager"]->GetPlugin(
                "Mailer")->GetTemplateList();
    }

    # load up possible values for SearchEngineUpdatePriority and
    #       RecommenderEngineUpdatePriority settings
    $PFactory = new PrivilegeFactory();
    $TaskPriorities = array(
            ApplicationFramework::PRIORITY_BACKGROUND => "Background",
            ApplicationFramework::PRIORITY_LOW => "Low",
            ApplicationFramework::PRIORITY_MEDIUM => "Medium",
            ApplicationFramework::PRIORITY_HIGH => "High");

    # load up possible values for DefaultSortField setting
    $ResourceSchema = new MetadataSchema(MetadataSchema::SCHEMAID_DEFAULT);
    $PossibleSortFields = array("R" => "Relevance");
    foreach ($ResourceSchema->GetFields(MetadataSchema::MDFTYPE_TEXT
            | MetadataSchema::MDFTYPE_NUMBER
            | MetadataSchema::MDFTYPE_DATE
            | MetadataSchema::MDFTYPE_TIMESTAMP
            | MetadataSchema::MDFTYPE_URL) as $FieldId => $Field)
    {
        if ($Field->IncludeInSortOptions())
        {
            $PossibleSortFields[$FieldId] = $Field->Name();
        }
    }

    # set up editing form
    $FormFields = array(
            "HEADING-General" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "General",
                    ),
            "PortalName" => array(
                    "Type" => FormUI::FTYPE_TEXT,
                    "Label" => "Portal Name",
                    "Required" => TRUE,
                    ),
            "AdminEmail" => array(
                    "Type" => FormUI::FTYPE_TEXT,
                    "Label" => "Administrator Email",
                    "Required" => TRUE,
                    "ValidateFunction" => array("FormUI", "ValidateEmail"),
                    ),
            "LegalNotice" => array(
                    "Type" => FormUI::FTYPE_TEXT,
                    "Label" => "Legal Notice",
                    ),
            "SiteKeywords" => array(
                    "Type" => FormUI::FTYPE_PARAGRAPH,
                    "Label" => "Site Keywords",
                    "Help" => "Used by search engines to find your site."
                            ." Separate words and phrases by commas.",
                    ),
            "AboutText" => array(
                    "Type" => FormUI::FTYPE_PARAGRAPH,
                    "Label" => "About Text",
                    "UseWYSIWYG" => TRUE,
                    ),
            # -------------------------------------------------
            "HEADING-Features" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Features",
                    ),
            "AnnouncementsEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Announcements",
                    ),
            "ForumsEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Forums",
                    ),
            "UserAgentsEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Saved Search Mailings",
                    ),
            "ResourceRatingsEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Resource Ratings/Recommendations",
                    ),
            "ResourceCommentsEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Resource Comments",
                    ),
            "IncrementalKeywordSearchEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Incremental Keyword Search",
                    ),
            "OAISQEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "OAI-SQ",
                    ),
            # -------------------------------------------------
            "HEADING-Interface" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Interface",
                    ),
            "NumAnnounceOnHomePage" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Announcements On Home Page",
                    "MinVal" => 1,
                    ),
            "NumResourcesOnHomePage" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Resources On Home Page",
                    "MinVal" => 0,
                    ),
            "ShowNumResourcesEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Display Resource Total On Home Page",
                    ),
            "DefaultActiveUI" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Default User Interface",
                    "Options" => $GLOBALS["AF"]->GetUserInterfaces(),
                    "Required" => TRUE,
                    ),
            "ForceDefaultActiveUI" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Set All Users to Default Interface",
                    "Help" => "When checked, this option will set all user"
                            ." accounts to the above Default User Interface.",
                    "Value" => FALSE,
                    ),
            "AllowMultipleUIsEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Allow Multiple User Interfaces",
                    ),
            "ResourceLaunchesNewWindowEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Resource Launches New Window",
                    ),
            "ShowGroupNamesEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Show Group Names In Full Record Page",
                    ),
            "ResourceTitleField" => array(
                    "Type" => FormUI::FTYPE_METADATAFIELD,
                    "Label" => "Resource Title Field",
                    "FieldTypes" => MetadataSchema::MDFTYPE_TEXT,
                    "SchemaId" => MetadataSchema::SCHEMAID_DEFAULT,
                    "Required" => TRUE,
                    ),
            "ResourceDescriptionField" => array(
                    "Type" => FormUI::FTYPE_METADATAFIELD,
                    "Label" => "Resource Description Field",
                    "FieldTypes" => MetadataSchema::MDFTYPE_PARAGRAPH,
                    "SchemaId" => MetadataSchema::SCHEMAID_DEFAULT,
                    ),
            "ResourceUrlField" => array(
                    "Type" => FormUI::FTYPE_METADATAFIELD,
                    "Label" => "Resource URL Field",
                    "FieldTypes" => MetadataSchema::MDFTYPE_URL,
                    "SchemaId" => MetadataSchema::SCHEMAID_DEFAULT,
                    ),
            "ResourceFileField" => array(
                    "Type" => FormUI::FTYPE_METADATAFIELD,
                    "Label" => "Resource File Field",
                    "FieldTypes" => MetadataSchema::MDFTYPE_FILE,
                    "SchemaId" => MetadataSchema::SCHEMAID_DEFAULT,
                    ),
            "ResourceScreenshotField" => array(
                    "Type" => FormUI::FTYPE_METADATAFIELD,
                    "Label" => "Resource Screenshot Field",
                    "FieldTypes" => MetadataSchema::MDFTYPE_IMAGE,
                    "SchemaId" => MetadataSchema::SCHEMAID_DEFAULT,
                    ),
            "PreferredLinkValue" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Preferred Link Value",
                    "Help" => "Used when both <i>Resource URL Field</i> and"
                            ." <i>Resource File Field</i> are set, for records"
                            ." where both fields have values.",
                    "Options" => array("URL" => "URL", "FILE" => "File"),
                    ),
            "TitlesLinkTo" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Titles Link to",
                    "Options" => array("URL" => "URL", "RECORD" => "Full Record"),
                    ),
            "CollapseMetadataFieldGroups" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Collapse Metadata Field Groups",
                    ),
            "RequireEmailWithFeedback" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Require Email Address with Feedback",
                    ),
            "ForumsAllowHTML" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Forums Allow HTML",
                    ),
            "ForumsUseWYSIWYG" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Forums Use WYSIWYG Editor for HTML",
                    ),
            # -------------------------------------------------
            "HEADING-Browsing" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Browsing",
                    ),
            "NumClassesPerBrowsePage" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Classifications Per Page When Browsing",
                    "MinVal" => 2,
                    ),
            "NumColumnsPerBrowsePage" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Columns Per Page When Browsing",
                    "MinVal" => 1,
                    "MaxVal" => 4,
                    "RecVal" => 2,
                    ),
            "BrowsingFieldId" => array(
                    "Type" => FormUI::FTYPE_METADATAFIELD,
                    "Label" => "Default Browsing Field",
                    "FieldTypes" => MetadataSchema::MDFTYPE_TREE,
                    "SchemaId" => MetadataSchema::SCHEMAID_DEFAULT,
                    "Required" => TRUE,
                    ),
            # -------------------------------------------------
            "HEADING-Search" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Search",
                    ),
            "SearchTermsRequired" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Default Search Term Handling",
                    "Options" => array(1 => "AND", 0 => "OR"),
                    ),
            "DisplayLimitsByDefault" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Display Search Limits by Default",
                    ),
            "DefaultRecordsPerPage" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Default Search Results Per Page",
                    "Required" => TRUE,
                    "MinVal" => 5,
                    "RecVal" => 10,
                    ),
            "MaxFacetsPerField" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Max Facets Per Field",
                    "Help" => "The maximum number of facets to display per field"
                            ." in faceted search.",
                    "Required" => TRUE,
                    "MinVal" => 2,
                    "MaxVal" => 100,
                    "RecVal" => 20,
                    ),
            "DefaultSortField" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Default Sort Field",
                    "Options" => $PossibleSortFields,
                    "Required" => TRUE,
                    ),
            # -------------------------------------------------
            "HEADING-Metrics" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Metrics",
                    ),
            "AddAWStatsScript" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "AWStats Logging",
                    ),
            # -------------------------------------------------
            "HEADING-System" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "System",
                    ),
            "MaxSimultaneousTasks" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Maximum Simultaneous Background Tasks",
                    "MinVal" => 1,
                    "MaxVal" => 32,
                    "Value" => $GLOBALS["AF"]->MaxTasks(),
                    ),
            "SearchEngineUpdatePriority" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Search Engine Update Task Priority",
                    "Options" => $TaskPriorities,
                    "Required" => TRUE,
                    ),
            "RecommenderEngineUpdatePriority" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Recommender Engine Update Task Priority",
                    "Options" => $TaskPriorities,
                    "Required" => TRUE,
                    ),
            # -------------------------------------------------
            "HEADING-Mailing" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Mailing",
                    ),
            "SavedSearchMailTemplate" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Saved Search Mail Template",
                    "Options" => $SavedSearchMailTemplates,
                    "Required" => TRUE,
                    ),
            # -------------------------------------------------
            "HEADING-UserAccounts" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "User Accounts",
                    ),
            "DefaultUserPrivs" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Default New User Privileges",
                    "AllowMultiple" => TRUE,
                    "Rows" => 15,
                    "Options" => $PFactory->GetPrivileges(TRUE, FALSE),
                    ),
            "SecureLogin" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Secure Login Form",
                    "Options" => array(
                            1 => "Enabled",
                            0 => "Optional",
                            -1 => "Disabled"),
                    ),
            );

    # return form field info to caller
    return $FormFields;
}

/**
* Load values for form fields.
* @param array $FormFields Associative array of form field parameters, in
*       the format expected by FormUI.
* @return array Associative array of form field parameters, in the format
*       expected by FormUI, with field values filled in where available.
*/
function LoadFormValues($FormFields)
{
    $ResourceSchema = new MetadataSchema(MetadataSchema::SCHEMAID_DEFAULT);
    foreach ($FormFields as $FieldName => $FieldParams)
    {
        if ($FieldParams["Type"] == FormUI::FTYPE_HEADING) {  continue;  }
        unset($FieldValue);
        switch ($FieldName)
        {
            case "ResourceTitleField":
            case "ResourceDescriptionField":
            case "ResourceUrlField":
            case "ResourceFileField":
            case "ResourceScreenshotField":
                # (strip off "Resource" and "Field" off of the form field name,
                #       and use the result for the standard metadata field name)
                $MFieldName = str_replace(
                        array("Resource", "Field"), "", $FieldName);
                $FieldValue = $ResourceSchema->StdNameToFieldMapping($MFieldName);
                break;

            case "PreferredLinkValue":
            case "TitlesLinkTo":
            case "CollapseMetadataFieldGroups":
            case "RequireEmailWithFeedback":
                $FieldValue = $GLOBALS["G_SysConfig"]->Value($FieldName);
                break;

            default:
                if (!isset($FormFields[$FieldName]["Value"]))
                {
                    # retrieve field values from SystemConfiguration where available
                    if (method_exists($GLOBALS["G_SysConfig"], $FieldName))
                    {
                        $FieldValue = $GLOBALS["G_SysConfig"]->$FieldName();
                    }
                    else
                    {
                        throw new Exception("Configuration setting for which value"
                                ." is not available (".$FieldName.").");
                    }
                }
                break;
        }
        if (isset($FieldValue))
        {
            $FormFields[$FieldName]["Value"] = $FieldValue;
        }
    }

    return $FormFields;
}

/**
* Save values from form fields.
* @param array $NewSettings Associative array with field names for the index
*       and field values for the values.
*/
function SaveFormValues($NewSettings)
{
    $ResourceSchema = new MetadataSchema(MetadataSchema::SCHEMAID_DEFAULT);
    foreach ($NewSettings as $FieldName => $NewFieldValue)
    {
        switch ($FieldName)
        {
            case "ForceDefaultActiveUI":
                if ($NewFieldValue)
                {
                    # set all users to selected user interface
                    $UFactory = new CWUserFactory();
                    foreach ($UFactory->GetUserIds() as $UserId)
                    {
                        $User = new CWUser($UserId);
                        $User->Set("ActiveUI", $NewSettings["DefaultActiveUI"]);
                    }
                }
                break;

            case "MaxSimultaneousTasks":
                $GLOBALS["AF"]->MaxTasks($NewFieldValue);
                break;

            case "ResourceDescriptionField":
            case "ResourceFileField":
            case "ResourceScreenshotField":
            case "ResourceTitleField":
            case "ResourceUrlField":
                if (!strlen($NewFieldValue))
                {
                    # (NULL is what you pass in to clear a standard field mapping)
                    $NewFieldValue = NULL;
                }
                # (strip off "Resource" and "Field" off of the form field name,
                #       and use the result for the standard metadata field name)
                $MFieldName = str_replace(
                        array("Resource", "Field"), "", $FieldName);
                $ResourceSchema->StdNameToFieldMapping($MFieldName, $NewFieldValue);
                break;

            case "CollapseMetadataFieldGroups":
            case "PreferredLinkValue":
            case "RequireEmailWithFeedback":
            case "TitlesLinkTo":
                # save values that match SystemConfiguration DB table columns
                $GLOBALS["G_SysConfig"]->Value($FieldName, $NewFieldValue);
                break;

            default:
                # save values via SystemConfiguration method if available
                if (method_exists($GLOBALS["G_SysConfig"], $FieldName))
                {
                    $GLOBALS["G_SysConfig"]->$FieldName($NewFieldValue);
                }
                else
                {
                    throw new Exception("New configuration value for which"
                            ." setting is not available (".$FieldName.").");
                }
                break;
        }
    }
}


# ----- MAIN -----------------------------------------------------------------

# check permissions
CheckAuthorization(PRIV_SYSADMIN, PRIV_COLLECTIONADMIN);

# set up form
$FormFields = DefineFormFields();

# load up current values for form fields
$FormFields = LoadFormValues($FormFields);

# instantiate form UI
$H_FormUI = new FormUI($FormFields);

# act on any button push
$ButtonPushed = GetFormValue("Submit");
switch ($ButtonPushed)
{
    case "Save":
        # check values and bail out if any are invalid
        if ($H_FormUI->ValidateFieldInput())
        {
            return;
        }

        # retrieve values from form
        $NewSettings = $H_FormUI->GetNewValuesFromForm();

        # save updated values
        SaveFormValues($NewSettings);

        # return to admin menu page
        $GLOBALS["AF"]->SetJumpToPage("SysAdmin");
        break;

    case "Cancel":
        # return to admin menu page
        $GLOBALS["AF"]->SetJumpToPage("SysAdmin");
        break;
}

