<?PHP
#
#   FILE:  AdvSysConfig.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

# ----- LOCAL FUNCTIONS ------------------------------------------------------

/**
* Define fields for form.
* @return array Associative array of form field parameters, in the format
*       expected by FormUI.
*/
function DefineFormFields()
{
    # load up possible values for DefaultCharacterSet
    $CharSets = array(
            "ISO-8859-1" => "ISO-8859-1",
            "UTF-8" => "UTF-8");

    # load up possible values for LoggingLevel
    $LoggingLevels = array(
        ApplicationFramework::LOGLVL_FATAL => "1 - Fatal",
        ApplicationFramework::LOGLVL_ERROR => "2 - Error",
        ApplicationFramework::LOGLVL_WARNING => "3 - Warning",
        ApplicationFramework::LOGLVL_INFO => "4 - Info",
        ApplicationFramework::LOGLVL_DEBUG => "5 - Debug",
        ApplicationFramework::LOGLVL_TRACE => "6 - Trace");

    # set up editing form
    $FormFields = array(
            "HEADING-Interface" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Interface",
                    ),
            "DefaultCharacterSet" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Default Character Encoding",
                    "Options" => $CharSets,
                    "Required" => TRUE,
                    ),
            # -------------------------------------------------
            "HEADING-Search" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Search",
                    ),
            "NumResourcesForSearchFacets" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Resources for Facet Generation",
                    "Units" => "resources",
                    "Help" => "The maximum number of resources to use when"
                            ." generating search facets.",
                    ),
            # -------------------------------------------------
            "HEADING-System" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "System",
                    ),
            "PageCacheEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Enable Page Caching",
                    "Help" => "Pages will be cached for anonymous users only.",
                    ),
            "PageCacheExpirationPeriod" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Page Cache Expiration Period",
                    "Units" => "minutes",
                    "MinVal" => 1,
                    ),
            "LoggingLevel" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Logging Level",
                    "Options" => $LoggingLevels,
                    ),
            "MaxExecutionTime" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Maximum Execution Time",
                    "Units" => "seconds",
                    "MinVal" => 5,
                    "RecVal" => 300,
                    ),
            "TemplateLocationCacheExpirationInterval" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Template Location Cache Expiration",
                    "Units" => "minutes",
                    "Help" => "Set to <i>0</i> to disable caching.",
                    "MinVal" => 0,
                    ),
            "ObjectLocationCacheExpirationInterval" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Object Location Cache Expiration",
                    "Units" => "minutes",
                    "Help" => "Set to <i>0</i> to disable caching.",
                    "MinVal" => 0,
                    ),
            "UseMinimizedJavascript" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Use Minimized JavaScript",
                    "Help" => "When enabled, minimized JavaScript files (file"
                            ." name ending in <i>.min.js</i>) will be searched"
                            ." for and used if found.",
                    ),
            "JavascriptMinimizationEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Generate Minimized JavaScript",
                    "Help" => "When enabled, the application framework will"
                            ." attempt to generate minimized JavaScript files.",
                    ),
            "UrlFingerprintingEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Add URL Fingerprints",
                    "Help" => "When enabled, the appplication framework will"
                            ." attempt to insert timestamp-based fingerprints"
                            ." into the names of CSS, JavaScript, and image"
                            ." files, to help with browser caching.",
                    ),
            "ScssSupportEnabled" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "SCSS Support Enabled",
                    "Help" => "When enabled, the application framework will"
                            ." look for SCSS versions of included CSS files,"
                            ." compile them into CSS, and use the resulting"
                            ." CSS files.",
                    ),
            "GenerateCompactCss" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Generate Compact CSS",
                    "Help" => "When enabled, any CSS generated from SCSS files"
                            ." will be compacted, to improve page access speeds.",
                    ),
            "LogSlowPageLoads" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Low Slow Page Loads",
                    ),
            "SlowPageLoadThreshold" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Slow Page Threshold",
                    "Units" => "seconds",
                    "Help" => "Pages that take longer than this will be logged"
                            ." if <i>Log Slow Page Loads</i> is enabled.",
                    "MinVal" => 2,
                    ),
            "LogHighMemoryUsage" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Low High Memory Usage",
                    ),
            "HighMemoryUsageThreshold" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "High Memory Usage Threshold",
                    "Units" => "%",
                    "Help" => "Pages that use more than this percentage of"
                            ." the PHP memory limit will be logged, if"
                            ." <i>Log High Memory Usage</i> is enabled.",
                    "MinVal" => 10,
                    "MaxVal" => 99,
                    "RecVal" => 90,
                    ),
            "PreferHttpHost" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Prefer HTTP_HOST",
                    "Help" => "If available, prefer <i>\$_SERVER[\"HTTP_HOST\"]</i>"
                            ." over <i>\$_SERVER[\"SERVER_NAME\"]</i> when"
                            ." determining the current URL.",
                    ),
            "RootUrlOverride" => array(
                    "Type" => FormUI::FTYPE_TEXT,
                    "Label" => "Root URL Override",
                    "Help" => "The \"root URL\" is the portion of the URL"
                            ." through the host name.  This setting primarily"
                            ." affects the values returned by the URL"
                            ." retrieval methods and the attempted insertion"
                            ." of clean URLs in outgoing HTML.",
                    "ValidateFunction" => array("FormUI", "ValidateUrl"),
                    ),
            # -------------------------------------------------
            "HEADING-PasswordRules" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Password Complexity Rules",
            ),
            "PasswordMinLength" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Minimum Password Length",
                    "Units" => "characters",
                    "MinVal" => 6,
            ),
            "PasswordUniqueChars" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "Minimum Unique Characters",
                    "Units" => "characters",
                    "MinVal" => 4,
            ),
            "PasswordRequiresPunctuation" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Require Punctuation Character",
            ),
            "PasswordRequiresMixedCase" => array(
                   "Type" => FormUI::FTYPE_FLAG,
                   "Label" => "Require Mixed-Case Passwords",
            ),
            "PasswordRequiresDigits" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Require Digits",
            ),
            # -------------------------------------------------
            "HEADING-Mailing" => array(
                    "Type" => FormUI::FTYPE_HEADING,
                    "Label" => "Mailing",
                    ),
            "MailingMethod" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Mailing Method",
                    "Options" => array(
                            Email::METHOD_PHPMAIL => "PHP mail()",
                            Email::METHOD_SMTP => "SMTP",
                            ),
                    ),
            "SmtpServer" => array(
                    "Type" => FormUI::FTYPE_TEXT,
                    "Label" => "SMTP Server",
                    "Help" => "Example: <i>mail.myhost.com</i>",
                    "DisplayIf" => array("MailingMethod" => EMail::METHOD_SMTP),
                    "ValidateFunction" => array("FormUI", "ValidateHostName"),
                    ),
            "SmtpPort" => array(
                    "Type" => FormUI::FTYPE_NUMBER,
                    "Label" => "SMTP Port",
                    "Help" => "Default: <i>25</i>",
                    "DisplayIf" => array("MailingMethod" => EMail::METHOD_SMTP),
                    "MinVal" => 1,
                    "MaxVal" => 65535,
                    "RecVal" => 25,
                    ),
            "UseAuthenticationForSmtp" => array(
                    "Type" => FormUI::FTYPE_FLAG,
                    "Label" => "Use Authentication for SMTP",
                    "DisplayIf" => array("MailingMethod" => EMail::METHOD_SMTP),
                    ),
            "SmtpUserName" => array(
                    "Type" => FormUI::FTYPE_TEXT,
                    "Label" => "SMTP User Name",
                    "Help" => "Only needed if using authentication for SMTP.",
                    "DisplayIf" => array(
                            "MailingMethod" => EMail::METHOD_SMTP,
                            "UseAuthenticationForSmtp" => TRUE),
                    ),
            "SmtpPassword" => array(
                    "Type" => FormUI::FTYPE_TEXT,
                    "Label" => "SMTP Password",
                    "Help" => "Only needed if using authentication for SMTP.",
                    "DisplayIf" => array(
                            "MailingMethod" => EMail::METHOD_SMTP,
                            "UseAuthenticationForSmtp" => TRUE),
                    ),
            "EmailLineEnding" => array(
                    "Type" => FormUI::FTYPE_OPTION,
                    "Label" => "Email Line Ending",
                    "Help" => "<i>CRLF</i> should be used whenever possible.",
                    "Options" => array(
                            "CRLF",
                            "CR",
                            "LF",
                            ),
                    ),
            );

    # return form field info to caller
    return $FormFields;
}

/**
* Load values for form fields.
* @param array $FormFields Associative array of form field parameters, in
*       the format expected by FormUI.
* @return array Associative array of form field parameters, in the format
*       expected by FormUI, with field values filled in where available.
*/
function LoadFormValues($FormFields)
{
    foreach ($FormFields as $FieldName => $FieldParams)
    {
        if ($FieldParams["Type"] == FormUI::FTYPE_HEADING) {  continue;  }
        unset($FieldValue);
        switch ($FieldName)
        {
            case "MailingMethod":
                $FieldValue = Email::DeliveryMethod();
                break;

            case "PageCacheExpirationPeriod":
                $FieldValue = $GLOBALS["AF"]->$FieldName() / 60;
                break;

            case "SmtpPassword":
                $FieldValue = Email::Password();
                break;

            case "SmtpPort":
                $FieldValue = Email::Port();
                break;

            case "SmtpServer":
                $FieldValue = Email::Server();
                break;

            case "SmtpUserName":
                $FieldValue = Email::UserName();
                break;

            case "UseAuthenticationForSmtp":
                $FieldValue = Email::UseAuthentication();
                break;

            case "GenerateCompactCss":
            case "HighMemoryUsageThreshold":
            case "JavascriptMinimizationEnabled":
            case "LoggingLevel":
            case "LogHighMemoryUsage":
            case "LogSlowPageLoads":
            case "MaxExecutionTime":
            case "ObjectLocationCacheExpirationInterval":
            case "PageCacheEnabled":
            case "ScssSupportEnabled":
            case "SlowPageLoadThreshold":
            case "TemplateLocationCacheExpirationInterval":
            case "UrlFingerprintingEnabled":
            case "UseMinimizedJavascript":
                $FieldValue = $GLOBALS["AF"]->$FieldName();
                break;

            default:
                if (!isset($FormFields[$FieldName]["Value"]))
                {
                    # retrieve field values from SystemConfiguration where available
                    if (method_exists($GLOBALS["G_SysConfig"], $FieldName))
                    {
                        $FieldValue = $GLOBALS["G_SysConfig"]->$FieldName();
                    }
                    else
                    {
                        throw new Exception("Configuration setting for which value"
                                ." is not available (".$FieldName.").");
                    }
                }
                break;
        }
        if (isset($FieldValue))
        {
            $FormFields[$FieldName]["Value"] = $FieldValue;
        }
    }

    return $FormFields;
}

/**
* Check new email delivery settings, logging errors to FormUI if found.
* @param array $NewValues New setting values, with field names for index.
* @return bool TRUE if problems were found with settings, otherwise FALSE.
*/
function ValidateEmailDeliverySettings($NewValues)
{
    # save new settings for testing
    Email::DeliveryMethod($NewValues["MailingMethod"]);
    Email::Password($NewValues["SmtpPassword"]);
    Email::Port($NewValues["SmtpPort"]);
    Email::Server($NewValues["SmtpServer"]);
    Email::UserName($NewValues["SmtpUserName"]);
    Email::UseAuthentication($NewValues["UseAuthenticationForSmtp"]);

    # if new settings do not test out okay
    if (!Email::DeliverySettingsOkay())
    {
        # log error messages
        $Errs = Email::DeliverySettingErrors();
        if (in_array("UseAuthentication", $Errs)
                || in_array("UserName", $Errs)
                || in_array("Password", $Errs))
        {
            $Msg = "Unable to connect with the specified <b>SMTP User Name</b>"
                    . " and <b>SMTP Password</b>. Please check that these"
                    . " values are correct to connect to <i>"
                    . $NewValues["SmtpServer"] . "</i>.";
            FormUI::LogError($Msg, "SmtpUserName");
            FormUI::LogError($Msg, "SmtpPassword");
        }
        elseif (in_array("Server", $Errs)
                || in_array("Port", $Errs))
        {
            $Msg = "An error was found with the <b>SMTP Server</b> or"
                    ." <b>SMTP Port</b> number.  Please check that these values"
                    ." are correct.";
            FormUI::LogError($Msg, "SmtpServer");
            FormUI::LogError($Msg, "SmtpPort");
        }
        elseif (in_array("TLS", $Errs))
        {
            $Msg = "An error was encountered trying to make a TLS connection to"
                    ." the specified server for SMTP.  Please check the server"
                    ." and port values to make sure they are correct.";
            FormUI::LogError($Msg, "SmtpServer");
        }
        else
        {
            $Msg = "An unknown error was encountered while trying to verify the"
                    ." <b>Mailing</b> settings.";
            FormUI::LogError($Msg);
        }

        # report to caller that problems were found
        return TRUE;
    }

    # report to caller that no problems were found
    return FALSE;
}

/**
* Save values from form fields.
* @param array $NewSettings Associative array with field names for the index
*       and field values for the values.
*/
function SaveFormValues($NewSettings)
{
    foreach ($NewSettings as $FieldName => $NewFieldValue)
    {
        switch ($FieldName)
        {
            case "MailingMethod":
                Email::DeliveryMethod($NewFieldValue);
                break;

            case "PageCacheExpirationPeriod":
                $GLOBALS["AF"]->$FieldName($NewFieldValue * 60);
                break;

            case "SmtpPassword":
                Email::Password($NewFieldValue);
                break;

            case "SmtpPort":
                Email::Port($NewFieldValue);
                break;

            case "SmtpServer":
                Email::Server($NewFieldValue);
                break;

            case "SmtpUserName":
                Email::UserName($NewFieldValue);
                break;

            case "UseAuthenticationForSmtp":
                Email::UseAuthentication($NewFieldValue);
                break;

            case "GenerateCompactCss":
            case "HighMemoryUsageThreshold":
            case "JavascriptMinimizationEnabled":
            case "LoggingLevel":
            case "LogHighMemoryUsage":
            case "LogSlowPageLoads":
            case "MaxExecutionTime":
            case "ObjectLocationCacheExpirationInterval":
            case "PageCacheEnabled":
            case "ScssSupportEnabled":
            case "SlowPageLoadThreshold":
            case "TemplateLocationCacheExpirationInterval":
            case "UrlFingerprintingEnabled":
            case "UseMinimizedJavascript":
                # save value via matching ApplicationFramework method
                $GLOBALS["AF"]->$FieldName($NewFieldValue, TRUE);
                break;

            default:
                # save values via SystemConfiguration method if available
                if (method_exists($GLOBALS["G_SysConfig"], $FieldName))
                {
                    $GLOBALS["G_SysConfig"]->$FieldName($NewFieldValue);
                }
                else
                {
                    throw new Exception("New configuration value for which"
                            ." setting is not available (".$FieldName.").");
                }
                break;
        }
    }

    # save email delivery settings that were set above
    $GLOBALS["G_SysConfig"]->Value(
            "EmailDeliverySettings", Email::DeliverySettings());
}


# ----- MAIN -----------------------------------------------------------------

# check permissions
CheckAuthorization(PRIV_SYSADMIN, PRIV_COLLECTIONADMIN);

# set up form
$FormFields = DefineFormFields();

# load up current values for form fields
$FormFields = LoadFormValues($FormFields);

# instantiate form UI
$H_FormUI = new FormUI($FormFields);

# act on any button push
$ButtonPushed = GetFormValue("Submit");
switch ($ButtonPushed)
{
    case "Save":
        # check values and bail out if any are invalid
        if ($H_FormUI->ValidateFieldInput())
        {
            return;
        }

        # retrieve values from form
        $NewSettings = $H_FormUI->GetNewValuesFromForm();

        # check mail delivery values and bail out if any are invalid
        if (ValidateEmailDeliverySettings($NewSettings))
        {
            return;
        }

        # save updated values
        SaveFormValues($NewSettings);

        # return to admin menu page
        $GLOBALS["AF"]->SetJumpToPage("SysAdmin");
        break;

    case "Cancel":
        # return to admin menu page
        $GLOBALS["AF"]->SetJumpToPage("SysAdmin");
        break;
}
