<?PHP
#
#   FILE:  PluginManager.php
#
#   Part of the ScoutLib application support library
#   Copyright 2009-2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu
#

/**
* Base class for all plugins.
*/
abstract class Plugin {

    # ----- PUBLIC INTERFACE -------------------------------------------------

    /**
    * Set the plugin attributes.  At minimum this method MUST set $this->Name
    * and $this->Version.  This is called when the plugin is loaded, and is
    * normally the only method called for disabled plugins (except for
    * SetUpConfigOptions(), which is called for pages within the plugin
    * configuration interface).
    */
    abstract function Register();

    /**
    * Set up plugin configuration options.  This is called if the plugin is
    * enabled and/or when loading the plugin configuration interface.  Config
    * options must be set up using this method (rather than going into
    * Register()) whenever their setup references data from outside of the
    * plugin in any fashion.
    * @return NULL if configuration setup succeeded, otherwise a string or
    *       array of strings containing error message(s) indicating why
    *       config setup failed.
    */
    function SetUpConfigOptions()
    {
        return NULL;
    }

    /**
    * Initialize the plugin.  This is called (if the plugin is enabled) after
    * all plugins have been loaded but before any methods for this plugin
    * (other than Register()) have been called.
    * @return NULL if initialization was successful, otherwise a string or
    *       array of strings containing error message(s) indicating why
    *       initialization failed.
    */
    function Initialize()
    {
        return NULL;
    }

    /**
    * Hook methods to be called when specific events occur.
    * For events declared by other plugins the name string should start with
    * the plugin base (class) name followed by "::" and then the event name.
    * @return Array of method names to hook indexed by the event constants
    *       or names to hook them to.
    */
    function HookEvents()
    {
        return array();
    }

    /**
    * Declare events defined by this plugin.  This is used when a plugin defines
    * new events that it signals or responds to.  Names of these events should
    * begin with the plugin base name, followed by "_EVENT_" and the event name
    * in all caps (for example "MyPlugin_EVENT_MY_EVENT").
    * @return Array with event names for the index and event types for the values.
    */
    function DeclareEvents()
    {
        return array();
    }

    /**
    * Perform any work needed when the plugin is first installed (for example,
    * creating database tables).
    * @return NULL if installation succeeded, otherwise a string containing
    *       an error message indicating why installation failed.
    */
    function Install()
    {
        return NULL;
    }

    /**
    * Perform any work needed when the plugin is upgraded to a new version
    * (for example, adding fields to database tables).
    * @param string $PreviousVersion The version number of this plugin that was
    *       previously installed.
    * @return NULL if upgrade succeeded, otherwise a string containing
    *       an error message indicating why upgrade failed.
    */
    function Upgrade($PreviousVersion)
    {
        return NULL;
    }

    /**
    * Perform any work needed when the plugin is uninstalled.
    * @return NULL if uninstall succeeded, otherwise a string containing
    *       an error message indicating why uninstall failed.
    */
    function Uninstall()
    {
        return NULL;
    }

    /**
    * Retrieve plugin information.
    * @return Array of attribute values indexed by attribute names.
    */
    final function GetAttributes()
    {
        return array(
                "Author" => $this->Author,
                "CfgPage" => $this->CfgPage,
                "CfgSetup" => $this->CfgSetup,
                "Description" => $this->Description,
                "Email" => $this->Email,
                "EnabledByDefault" => $this->EnabledByDefault,
                "InitializeAfter" => is_array($this->InitializeAfter)
                        ? $this->InitializeAfter : array($this->InitializeAfter),
                "InitializeBefore" => is_array($this->InitializeBefore)
                        ? $this->InitializeBefore : array($this->InitializeBefore),
                "Instructions" => $this->Instructions,
                "Name" => $this->Name,
                "Requires" => $this->Requires,
                "Url" => $this->Url,
                "Version" => $this->Version,
                );
    }

    /**
    * Get/set plugin configuration setting.
    * @param string $SettingName Name of configuration value.
    * @param mixed $NewValue New setting value.
    * @return Requested value, or NULL if value was not set or there was no
    *       configuration value with the specified name.
    */
    final function ConfigSetting($SettingName, $NewValue = NULL)
    {
        # if a new value was supplied for the setting
        if (func_num_args() > 1)
        {
            # if this setting has a filter function specified
            if (array_key_exists($SettingName, $this->CfgSetup)
                    && array_key_exists("SettingFilter",
                            $this->CfgSetup[$SettingName]))
            {
                # pass new value through filter function
                $FilterMethod = $this->CfgSetup[$SettingName]["SettingFilter"];
                $NewValue = $this->$FilterMethod($SettingName, $NewValue);
            }

            # if caller requested that setting be cleared
            if ($NewValue === NULL)
            {
                # clear setting
                unset($this->Cfg[$SettingName]);
            }
            else
            {
                # save new value for setting
                $this->Cfg[$SettingName] = $NewValue;
            }

            # if we have a way of saving configuration settings
            if (is_callable($this->CfgSaveCallback))
            {
                # save new configuration settings
                call_user_func_array($this->CfgSaveCallback,
                        array(get_class($this), $this->Cfg));
            }
        }

        # return current value of setting to caller
        return isset($this->Cfg[$SettingName]) ? $this->Cfg[$SettingName] : NULL;
    }


    # ----- PROTECTED INTERFACE ----------------------------------------------

    /** Name of the plugin's author. */
    protected $Author = NULL;
    /** Text description of the plugin. */
    protected $Description = NULL;
    /** Contact email for the plugin's author. */
    protected $Email = NULL;
    /** Whether the plugin should be enabled by default when installed. */
    protected $EnabledByDefault = FALSE;
    /** Plugins that should be initialized after us. */
    protected $InitializeBefore = array();
    /** Plugins that should be initialized before us. */
    protected $InitializeAfter = array();
    /** Instructions for configuring the plugin (displayed on the
            automatically-generated configuration page if configuration
            values are supplied). */
    protected $Instructions = NULL;
    /** Proper (human-readable) name of plugin. */
    protected $Name = NULL;
    /** Version number of plugin in the format X.X.X (for example: 1.2.12). */
    protected $Version = NULL;
    /** Web address for more information about the plugin. */
    protected $Url = NULL;

    /**
    * Array with plugin base (class) names for the index and minimum version
    * numbers for the values.  Special indexes of "PHP" may be used to
    * specify a minimum required PHP version or "PHPX_xxx" to specify a required
    * PHP extension, where "xxx" is the extension name (e.g. "PHPX_GD").  The
    * version number value is ignored for PHP extensions.
    */
    protected $Requires = array();

    /**
    * Associative array describing the configuration values for the plugin.
    * The first index is the name of the configuration setting, and the second
    * indicates the type of information about that setting.  For more information
    * please see <a href="implementingplugins.html">Implementing
    * CWIS Plugins</a>.
    */
    protected $CfgSetup = array();

    /**
    * Name of configuration page for plugin.
    */
    protected $CfgPage = NULL;


    # ----- PRIVATE INTERFACE ------------------------------------------------

    /** Plugin configuration values. */
    private $Cfg;
    /** Callback to invoke when configuration value changes. */
    private $CfgSaveCallback;

    /** @cond */
    /**
    * Set all configuration values (only for use by PluginManager).
    * @param NewValues Array of new configuration values.
    */
    final public function SetAllCfg($NewValues)
    {
        $this->Cfg = $NewValues;
    }
    /** @endcond */

    /** @cond */
    /**
    * Set callback to be invoked when configuration values change (only for use
    *       by PluginManager)..
    * @param Callback Callback function or method.
    */
    final public function SetCfgSaveCallback($Callback)
    {
        $this->CfgSaveCallback = $Callback;
    }
    /** @endcond */
}


