<?PHP

class POAIItem implements OAIItem {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function POAIItem($ItemId, array $RepDescr, $SearchInfo = NULL)
    {
        # save ID for later use
        $this->Id = $ItemId;

        # save any search info supplied
        $this->SearchInfo = $SearchInfo;

        # save the repository description
        $this->RepDescr = $RepDescr;

        # attempt to create resource object
        $this->Resource = new Resource($ItemId);

        # if resource object creation failed
        if ($this->Resource->Status() == -1)
        {
            # set status to -1 to indicate constructor failure
            $this->LastStatus = -1;
        }
        else
        {
            # set status to 1 to indicate constructor success
            $this->LastStatus = 1;

            # if cumulative rating data is available for this resource
            global $SysConfig;
            if ($SysConfig->ResourceRatingsEnabled()
                    && $this->Resource->CumulativeRating())
            {
                # add cumulative rating data to search info
                $this->SearchInfo["cumulativeRating"] =
                        $this->Resource->CumulativeRating();
                $this->SearchInfo["cumulativeRatingScale"] = 100;
            }
        }
    }

    function GetId() {  return $this->Id;  }

    function GetDatestamp()
    {
        $DateString = $this->Resource->Get("Date Of Record Creation");
        if (!isset($DateString) ||
            $DateString == "0000-00-00 00:00:00") {  $DateString = date("Y-m-d");  }
        $Date = new Date($DateString);
        return $Date->FormattedISO8601();
    }

    function GetValue($ElementName)
    {
        # if requested value is a preferred link value
        if ($ElementName == -3)
        {
            $ReturnValue = $this->GetPreferredLinkValueForResource($this->Resource);
        }
        # if requested value is full record page URL
        elseif ($ElementName == -2)
        {
            # set value to full record page URL
            $ReturnValue = $this->GetFullRecordUrlForResource($this->Resource->Id());
        }
        # if requested value is fixed default
        elseif ($ElementName == -3)
        {
            $ReturnValue = NULL;
        }
        else
        {
            # retrieve value
            $ReturnValue = $this->Resource->GetByFieldId($ElementName);

            # this isn't technically necessary for the checks below, but it
            # reduces some overhead when the field obviously isn't a reference
            if (is_array($ReturnValue))
            {
                $Schema = new MetadataSchema();
                $Field = $Schema->GetField($ElementName);

                # if the field is a reference field
                if ($Field->Type() == MetadataSchema::MDFTYPE_REFERENCE)
                {
                    # translate each resource ID to an OAI identifier
                    foreach ($ReturnValue as $Key => $Value)
                    {
                        $ReturnValue[$Key] = $this->GetOaiIdentifierForResource(
                            $Value);
                    }
                }
            }

            # strip out any HTML tags if text value
            if (is_string($ReturnValue))
            {
                $ReturnValue = strip_tags($ReturnValue);
            }

            # format correctly if standardized date
            if ($this->GetQualifier($ElementName) == "W3C-DTF")
            {
                $Timestamp = strtotime($ReturnValue);
                $ReturnValue = date('Y-m-d\TH:i:s', $Timestamp)
                        .substr_replace(date('O', $Timestamp), ':', 3, 0);
            }
        }

        # return value to caller
        return $ReturnValue;
    }

    function GetQualifier($ElementName)
    {
        $ReturnValue = NULL;
        $Qualifier = $this->Resource->GetQualifierByFieldId($ElementName, TRUE);
        if (is_array($Qualifier))
        {
            foreach ($Qualifier as $ItemId => $QualObj)
            {
                if (is_object($QualObj))
                {
                    $ReturnValue[$ItemId] = $QualObj->Name();
                }
            }
        }
        else
        {
            if (isset($Qualifier) && is_object($Qualifier))
            {
                $ReturnValue = $Qualifier->Name();
            }
        }
        return $ReturnValue;
    }

    function GetSets()
    {
        # start out with empty list
        $Sets = array();

        # for each possible metadata field
        $Schema = new MetadataSchema();
        $Fields = $Schema->GetFields(MetadataSchema::MDFTYPE_TREE
                |MetadataSchema::MDFTYPE_CONTROLLEDNAME
                |MetadataSchema::MDFTYPE_OPTION);
        foreach ($Fields as $Field)
        {
            # if field is flagged for use for OAI sets
            if ($Field->UseForOaiSets())
            {
                # retrieve values for resource for this field and add to set list
                $FieldName = $Field->Name();
                $Values = $this->Resource->Get($FieldName);
                if (isset($Values) && ($Values != NULL))
                {
                    $NormalizedFieldName = $this->NormalizeForSetSpec($FieldName);
                    if (is_array($Values) && count($Values))
                    {
                        foreach ($Values as $Value)
                        {
                            $Sets[] = $NormalizedFieldName.":"
                                    .$this->NormalizeForSetSpec($Value);
                        }
                    }
                    else
                    {
                        $Sets[] = $NormalizedFieldName.":"
                                .$this->NormalizeForSetSpec($Values);
                    }
                }
            }
        }

        # return list of sets to caller
        return $Sets;
    }

    function GetSearchInfo()
    {
        return $this->SearchInfo;
    }

    function Status()
    {
        return $this->LastStatus;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

    var $Id;
    var $Resource;
    var $LastStatus;
    var $RepDescr;
    var $SearchInfo;

    # normalize value for use as an OAI set spec
    function NormalizeForSetSpec($Name)
    {
        return preg_replace("/[^a-zA-Z0-9\-_.!~*'()]/", "", $Name);
    }

    /**
     * Get the URL to the full record of a resource.
     * @param int $ResourceId resource ID
     * @return the URL to the full record of the resource
     * @see GetOaiIdentifierForResource()
     */
    protected function GetFullRecordUrlForResource($ResourceId)
    {
        $SafeResourceId = urlencode($ResourceId);
        return OurBaseUrl() .
            $GLOBALS["AF"]->GetCleanUrlForPath(
            "index.php?P=FullRecord&ID=" . $SafeResourceId );
    }

    /**
    * Get the Preferred Link Value for a resource.
    * @param Resource $Resource
    * @return URL corresponding to the pref. link value.
    */
    protected function GetPreferredLinkValueForResource($Resource)
    {
        global $G_SysConfig;

        $GLOBALS["AF"]->LoadFunction("GetResourceFieldValue");

        # start off assuming no result
        $Result = "";

        # pull the mapped URL and File fields from the schema
        $Schema = new MetadataSchema();
        $UrlField = $Schema->GetFieldByMappedName("Url");
        $FileField = $Schema->GetFieldByMappedName("File");

        $Url = ( !is_null($UrlField) &&
                 $UrlField->Status() === MetadataSchema::MDFSTAT_OK )
            ? GetResourceFieldValue($Resource, $UrlField) : "";
        $Files = ( !is_null($FileField) &&
                   $FileField->Status() === MetadataSchema::MDFSTAT_OK )
            ? GetResourceFieldValue($Resource, $FileField) : array();

        # figure out what the preferred link should be
        if (is_array($Files) && count($Files) > 0
            && ($G_SysConfig->Value("PreferredLinkValue") == "FILE"
                || !$G_SysConfig->Value("PreferredLinkValue")))
        {
            # if we prefer files, use the first one
            $LinkFlie = array_shift($Files);
            $Result =  OurBaseUrl() .
                $GLOBALS["AF"]->GetCleanUrlForPath(
                $LinkFile->GetLink() );
        }
        elseif (strlen($Url) > 0)
        {
            # otherwise, if there's a sane-looking URL, use that
            if ( preg_match('/^\s*[a-zA-Z]+:\/\//', $Url) )
            {
                $Result = $Url;
            }
        }

        return $Result;
    }

    /**
     * Get the OAI identifier of a resource.
     * @param int $ResourceId resource ID
     * @return the OAI identifier of a resource
     * @see GetFullRecordUrlForResource()
     */
    protected function GetOaiIdentifierForResource($ResourceId)
    {
        # return encoded value to caller
        return "oai:".$this->RepDescr["IDDomain"]
                .":".$this->RepDescr["IDPrefix"]."-".$ResourceId;
    }

}


?>
